#include "formatutils.h"


#include <QTextCodec>
#include <QTextEncoder>
#include <QDataStream>

QString FormatUtils::dtToFullIsoStr(const QDateTime &dt)
{
    if(!dt.isValid() || dt.isNull()) return QString();
    return QDateTime::fromString(
                dt.toString(Qt::RFC2822Date), Qt::RFC2822Date)
            .toString(Qt::ISODate);
}

QDateTime FormatUtils::strToDt(const QString &dt)
{
    if(dt.isEmpty()) return QDateTime();
    QDateTime res = QDateTime::fromString(dt, Qt::ISODateWithMs);
    if(!res.isValid())
    {
        res =  QDateTime::fromString(dt, Qt::ISODate);
    }
    if(!res.isValid())
    {
        res =  QDateTime::fromString(dt, Qt::RFC2822Date);
    }
    if(!res.isValid())
    {
        res =  QDateTime::fromString(dt, Qt::TextDate);
    }
    if(!res.isValid())
    {
        res =  QDateTime::fromString(dt, "yyyyMMddThh:mm:ss");
    }
    if(!res.isValid())
    {
        res =  QDateTime::fromString(dt, "dd.MM.yyyy hh:mm:ss");
    }
    if(!res.isValid())
    {
        res =  QDateTime::fromString(dt, "dd.MM.yyyy hh:mm");
    }
    return res;
}

FixNumber FormatUtils::parseSumm(const QString &name, const QVariantMap &map, quint32 decPoint, CoreApiConst::ErrorCode *error)
{
    if(error) *error = CoreApiConst::ErrorCode::Ok;
    FixNumber res;
    if(map.contains(name))
    {
        QVariant val = map[name];
        if(val.type() == QVariant::Map)
        {
            res.setMap(val.toMap());
            if(res.decimals() > 0 && decPoint > 0)
            {
                if(res.decimals() != static_cast<qint32>(decPoint))
                {
                    return  parseSumm(res.toString(), decPoint, error);
                }
            }
            return res;
        }
        else if(val.type() == QVariant::String)
        {
            return parseSumm(val.toString().trimmed().replace(",", "."), decPoint, error);
        }
    }
    else
    {
    }
    if(error) *error = CoreApiConst::ErrorCode::InvalidSumFormat;
    return res;
}

FixNumber FormatUtils::parseSumm(const QString &sum, quint32 decPoint, CoreApiConst::ErrorCode *error)
{
    if(error) *error = CoreApiConst::ErrorCode::InvalidSumFormat;
    FixNumber res;
    QString s = sum.trimmed().replace(",", ".");
    bool ok = false;
    double vd = s.toDouble(&ok);
    if(ok)
    {
        if(s.contains("."))
        {
            if(s.startsWith(".")) s = "0" + s;
            quint32 dp = static_cast<quint32>(s.mid(s.indexOf(".") + 1).size());
            if( dp < decPoint)
            {
                s += QString(static_cast<int>(decPoint - dp), QLatin1Char('0'));
            }
            res.setDecimal(static_cast<qint32>(decPoint));
            if(dp <= decPoint)
            {
                res.setValue(s.remove(".").toLongLong());
            }
            else
            {
                res.setDouble(vd);
            }
        }
        else
        {
            res.setDecimal(static_cast<qint32>(decPoint));
            res.setDouble(vd);
        }
        if(error) *error = CoreApiConst::ErrorCode::Ok;
        return res;
    }
//    lmWarning() << ok << vd << res.toString();
    return res;
}

QVariantMap FormatUtils::unitMaps(const QVariantMap &first, const QVariantMap &second)
{
    QVariantMap res = first;
    for(auto it = second.constBegin(); it != second.constEnd(); ++it)
    {
        if(!res.contains(it.key())) res.insert(it.key(), it.value());
    }
    return res;
}

bool FormatUtils::checkStringForFs(const QString &value)
{
    QTextCodec *codec = QTextCodec::codecForName("CP866");
    QString str = codec->toUnicode(codec->fromUnicode(value));
    return str == value;
}

QString FormatUtils::formatStringForFs(const QString &value)
{
    static const QMap<QString, QString> REPLACEMENT = {
        {"\ufeff", ""},
        {"\uffef", ""},
        {"\u200b", ""},
        {"\u00a0", " "},
        {"\u00ad", " "},
        {"\u200d", ""},
        {"\u2060", " "},
        {"\u1680", " "},
        {"\u037e", ";"},
        {"\u202d", ""},
        {"\u202e", ""},
        {"\ua4f8", "."},
        {"\ua4f9", ","},
        {"\ua4fc", ";"},
        {"\u00ab", "\""},
        {"\u00bb", "\""},
        {"\u3003", "\""},
        {"\u030f", "\""},
        {"\u030b", "\""},
        {"\u02dd", "\""},
        {"\u1fce", "\""},
        {"\u2212", "-"},
        {"\u2010", "-"},
        {"\u2f00", "-"},
        {"\u0304", "-"},
        {"\u00af", "-"},
        {"\u02c9", "-"},
        {"\u058a", "-"},
        {"\u0000", ""},
    };
    QString val = value;
    for(auto it = REPLACEMENT.constBegin(); it != REPLACEMENT.constEnd(); ++it)
    {
        val = val.replace(it.key(), it.value());
    }
    QTextCodec *codec = QTextCodec::codecForName("CP866");
    QTextEncoder *e = codec->makeEncoder(QTextCodec::ConvertInvalidToNull);
    QString str = codec->toUnicode(e->fromUnicode(val));
    delete e;
    return str;
}

FixNumber FormatUtils::uint48ToFixNumber(const QByteArray &data, quint32 mult, bool *ok)
{
    if(ok) *ok = false;
    if(data.size() != 6) return FixNumber(mult, -1ll);
    QByteArray buf = data + QByteArray(2, '\x00');
    QDataStream ds(buf);
    ds.setByteOrder(QDataStream::LittleEndian);
    qint64 val = 0;
    ds >> val;
    if(ok) *ok = true;
    return FixNumber(mult, val);
}

QByteArray FormatUtils::fixNumberToUint48(const FixNumber &val, bool *ok)
{
    if(ok) *ok = false;
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << val.value();
    if(buf.size() != 8 || buf[6] != '\x00' || buf[7] != '\x00') return QByteArray();
    if(ok) *ok = true;
    return buf.mid(0, 6);
}

FixNumber FormatUtils::uint40ToFixNumber(const QByteArray &data, quint32 mult, bool *ok)
{
    if(ok) *ok = false;
    if(data.size() != 5) return FixNumber(mult, -1ll);
    QByteArray buf = data + QByteArray(3, '\x00');
    QDataStream ds(buf);
    ds.setByteOrder(QDataStream::LittleEndian);
    qint64 val = 0;
    ds >> val;
    if(ok) *ok = true;
    return FixNumber(mult, val);
}

QByteArray FormatUtils::fixNumberToUint40(const FixNumber &val, bool *ok)
{
    if(ok) *ok = false;
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << val.value();
    if(buf.size() != 8 || buf[5] != '\x00' || buf[6] != '\x00'|| buf[7] != '\x00') return QByteArray();
    if(ok) *ok = true;
    return buf.mid(0, 5);
}

quint32 FormatUtils::fsToUint32(const QByteArray &data, bool *ok)
{
    if(ok) *ok = false;
    if(data.size() != 4) return 0;
    QDataStream ds(data);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint32 val = 0;
    ds >> val;
    if(ok) *ok = true;
    return val;
}

QByteArray FormatUtils::uint32ToFs(quint32 val)
{
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << val;
    return buf;
}

quint32 FormatUtils::fsToUint16(const QByteArray &data, bool *ok)
{
    if(ok) *ok = false;
    if(data.size() != 2) return 0;
    QDataStream ds(data);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint16 val = 0;
    ds >> val;
    if(ok) *ok = true;
    return val;
}

QByteArray FormatUtils::uint16ToFs(quint16 val)
{
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << val;
    return buf;
}

bool FormatUtils::checkString(const QString &val, qint32 maxLen, bool fixedLen, bool canBeEmpty, QString &suffixMsg)
{
//    lmWarning() << val << maxLen << fixedLen << canBeEmpty << logtab
//                << val.isEmpty() << val.size() << FormatUtils::checkStringForFs(val);
    if(val.isEmpty())
    {
        if(!canBeEmpty) suffixMsg = QStringLiteral(" не может быть пустым");
        return canBeEmpty;
    }
    if(val.size() > maxLen)
    {
        suffixMsg = QStringLiteral(" не может содержать больше %1 символов").arg(maxLen);
        return false;
    }
    if(fixedLen && val.size() != maxLen)
    {
        suffixMsg = QStringLiteral(" должно содержать в точности %1 cимволов").arg(maxLen);
        return false;
    }
    if(!FormatUtils::checkStringForFs(val))
    {
        suffixMsg = QStringLiteral(" содержит недопустимые символы.");
        return false;
    }
    return true;
}

bool FormatUtils::checkEmail(const QString &val, qint32 maxLen, bool canBeEmpty, QString &suffixMsg)
{
    if(!checkString(val, maxLen, false, canBeEmpty, suffixMsg)) return false;
    if(!val.isEmpty() && (!val.contains("@") || !val.contains(".")))
    {
        suffixMsg = QStringLiteral(" имеет некорректный формат");
        return false;
    }
    return true;
}

bool FormatUtils::checkPhone(const QString &val, qint32 maxLen, bool canBeEmpty, QString &suffixMsg)
{
    if(canBeEmpty && val.isEmpty()) return true;
    if(!checkString(val, maxLen, false, canBeEmpty, suffixMsg)) return false;
    if(!val.startsWith("+"))
    {
        suffixMsg = QStringLiteral(" должен начинаться с '+'");
        return false;
    }
    QString s = val;
    s = s.remove("+").remove("-").remove("(").remove(")").trimmed();
    bool ok = false;
    if(s.isEmpty() || s.toLongLong(&ok) <= 0 || !ok)
    {
        suffixMsg = QStringLiteral(" имеет некорректное значение ") + val;
        return false;
    }
    return true;
}

QString FormatUtils::rawLabelToTransport(const QString &text)
{
    return text; //QString::fromUtf8(text.toUtf8().toBase64());
}

QString FormatUtils::rawLabelFromTransport(const QString &text)
{
    QString newLabel = text;
    if(newLabel.startsWith("\u001d"))newLabel = newLabel.mid(1);
    return newLabel;// QString::fromUtf8(QByteArray::fromBase64(text.toUtf8()));
}

FormatUtils::FormatUtils()
{

}


FormatUtils::~FormatUtils()
{

}
