#include "labelcode.h"

#include "productcodecreator.h"

#include "formatutils.h"
#include "cashboxbuildconfig.h"
#include "crc.h"
#include <cmath>
#include <QSet>


static const QSet<qint64> FIXED_EAN13_MARKS = {
    2400001323807ll,
    2400003675805ll,
    2400001807703ll,
    2400001818303ll,
    2400002186203ll,
    2400001368105ll,
    2400001225408ll,
    2400001225606ll,
    2400001226108ll,
    2400001393503ll,
    2400001858309ll,
    2400001858507ll,
    2400002052805ll,
    2400002052904ll,
    2400002984502ll,
    2400003117107ll,
    2400003117206ll,
    2400003207907ll,
    2400003215308ll,
    2400003297700ll,
    2400003356704ll,
    2400003356803ll,
    2400003433108ll,
    2400003492303ll,
    2400003495700ll,
    2400003495809ll,
    2400003495908ll,
    2400003496004ll,
    2400003496103ll,
    2400001226306ll,
    2400001226405ll,
    2400001393107ll,
    2400001393602ll,
    2400001565306ll,
    2400001857203ll,
    2400001857005ll,
    2400002015909ll,
    2400002016005ll,
    2400002016104ll,
    2400003161209ll,
    2400003227806ll,
    2400003237409ll,
    2400003263408ll,
    2400003356902ll,
    2400003356902ll,
    2400002886806ll,
    2400002886707ll,
};


LabelCode::LabelCode()
    : tradeMark_()
    , code_()
    , t91_()
    , t92_()
    , t93_()
{

}


LabelCode::LabelCode(const LabelCode &other)
    : tradeMark_(other.tradeMark_)
    , code_(other.code_)
    , t91_(other.t91_)
    , t92_(other.t92_)
    , t93_(other.t93_)
{

}

LabelCode::LabelCode(LabelCode &&other)
    : tradeMark_()
    , code_(other.code_)
    , t91_()
    , t92_()
    , t93_()

{
    tradeMark_.swap(other.tradeMark_);
    t91_.swap(other.t91_);
    t92_.swap(other.t92_);
    t93_.swap(other.t93_);
}

LabelCode::~LabelCode()
{

}

bool LabelCode::readyToB() const
{
    if(tradeMark_.isEmpty()) return false;
    if(code_.type() == ProductCode::Type::LabelGs1M)
    {
        return (idx91() > 0 && idx92() > 0) || t93_.size() == 4;
    }
    return true;
}

QByteArray LabelCode::preparedLabel() const
{
    if(tradeMark_.isEmpty()) return QByteArray();
    QByteArray res = tradeMark_.toLatin1();
    while(res.endsWith('\x00')) res = res.mid(0, res.size() - 1);
    return res;
}

QString LabelCode::tradeMark() const
{
    return tradeMark_;
}

ProductCode LabelCode::productCode() const
{
    return code_;
}

ProductCode::Type LabelCode::type() const
{
    return code_.type();
}

QString LabelCode::typeName() const
{
    switch (type()) {
    case ProductCode::Type::LabelEan8     : return "КТ EAN-8";
    case ProductCode::Type::LabelEan13    : return "КТ EAN-13";
    case ProductCode::Type::LabelItf14    : return "КТ ITF-14";
    case ProductCode::Type::LabelGs10     : return "КТ GS1.0";
    case ProductCode::Type::LabelGs1M     : return "КТ GS1.M";
    case ProductCode::Type::LabelShortCode: return "КТ КМК";
    case ProductCode::Type::LabelFur      : return "КТ МИ";
    case ProductCode::Type::LabelEgais2   : return "КТ ЕГАИС-2.0";
    case ProductCode::Type::LabelEGais3   : return "КТ ЕГАИС-3.0";
    case ProductCode::Type::LabelF1       : return "КТ Ф.1";
    case ProductCode::Type::LabelF2       : return "КТ Ф.2";
    case ProductCode::Type::LabelF3       : return "КТ Ф.3";
    case ProductCode::Type::LabelF4       : return "КТ Ф.4";
    case ProductCode::Type::LabelF5       : return "КТ Ф.5";
    case ProductCode::Type::LabelF6       : return "КТ Ф.6";
    default: return "КТ Н";
    }
}

const QString &LabelCode::t91() const
{
    return t91_;
}

const QString &LabelCode::t92() const
{
    return t92_;
}

const QString &LabelCode::t93() const
{
    return t93_;
}

quint8 LabelCode::idx91() const
{
    if(tradeMark_.isEmpty() || t91_.isEmpty()) return 0;
    QByteArray ba = preparedLabel();
    QByteArray ba91 = "91" + t91_.toLatin1();
//    lmWarning() << ba.indexOf(ba91) << QString::number(ba.indexOf(ba91), 16) << logtab << logbinary(ba) << logtab << logbinary(ba91) ;

    return static_cast<quint8>(ba.indexOf(ba91));
}

quint8 LabelCode::idx92() const
{
    if(tradeMark_.isEmpty() || t92_.isEmpty()) return 0;
    QByteArray ba = preparedLabel();
    QByteArray ba92 = "92" + t92_.toLatin1();
//    lmWarning() << ba.indexOf(ba92) << QString::number(ba.indexOf(ba92), 16) << logtab << logbinary(ba) << logtab << logbinary(ba92) ;
    return static_cast<quint8>(ba.indexOf(ba92));

}

QString LabelCode::calc2115() const
{
    Crc32 crc;
    quint32 v = crc(preparedLabel());
    QString res = QString("%1").arg(v, 10, 10, QLatin1Char('0'));
    res = res.mid(res.size() - 4);
    return res;
}

fdf::LabelCodeType LabelCode::lct2100() const
{
    fdf::LabelCodeType res = fdf::LabelCodeType::Unknown;
    if(code_.isGsM())
    {
        if(code_.type() == ProductCode::Type::LabelShortCode) res = fdf::LabelCodeType::Short;
        else if(code_.type() == ProductCode::Type::LabelGs1M)
        {
            if(t92_.length() == 88) res = fdf::LabelCodeType::Code88Checked;
            else if(t92_.length() == 44)
            {
                quint32 crc1 = tailCrc(), crc2 = 0;
                QDataStream ds(QByteArray::fromBase64(t92_.toLatin1()).mid(28, 4));
                ds.setByteOrder(QDataStream::LittleEndian);
                ds >> crc2;
                if(crc1 == crc2) res = fdf::LabelCodeType::Code44NotChecked;
                else res = fdf::LabelCodeType::Code44Checked;
            }
            else if(t93_.length() == 4)
            {
                res = fdf::LabelCodeType::Code4NotChecked;
            }
        }
    }
    return res;
}


bool LabelCode::isValid() const
{
    return !tradeMark_.isEmpty() && code_.isValid();
}

bool LabelCode::isEncodedTradeMark() const
{
    return code_.type() ==     ProductCode::Type::LabelUnknown ||
            code_.type() ==   ProductCode::Type::LabelEan8  ||
            code_.type() ==   ProductCode::Type::LabelEan13 ||
            code_.type() ==   ProductCode::Type::LabelItf14 ||
            code_.type() ==   ProductCode::Type::LabelGs10      ||
            code_.type() ==    ProductCode::Type::LabelGs1M ||
            code_.type() ==    ProductCode::Type::LabelShortCode ||
            code_.type() ==   ProductCode::Type::LabelFur       ||
            code_.type() ==   ProductCode::Type::LabelEgais2    ||
            code_.type() ==   ProductCode::Type::LabelEGais3    ||
            code_.type() ==   ProductCode::Type::LabelF1||
            code_.type() ==   ProductCode::Type::LabelF2||
            code_.type() ==   ProductCode::Type::LabelF3||
            code_.type() ==   ProductCode::Type::LabelF4||
            code_.type() ==   ProductCode::Type::LabelF5||
            code_.type() ==   ProductCode::Type::LabelF6;
}

bool LabelCode::isGs() const
{
    return code_.type() ==    ProductCode::Type::LabelGs1M ||
            code_.type() ==    ProductCode::Type::LabelShortCode;
}

bool LabelCode::isRecognizedCode() const
{
    return  code_.type() ==   ProductCode::Type::LabelEan8      ||
            code_.type() ==   ProductCode::Type::LabelEan13     ||
            code_.type() ==   ProductCode::Type::LabelItf14     ||
            code_.type() ==   ProductCode::Type::LabelGs10      ||
            code_.type() ==   ProductCode::Type::LabelGs1M      ||
            code_.type() ==   ProductCode::Type::LabelShortCode ||
            code_.type() ==   ProductCode::Type::LabelFur       ||
            code_.type() ==   ProductCode::Type::LabelEgais2    ||
            code_.type() ==   ProductCode::Type::LabelEGais3    ;
}

QStringList LabelCode::searchCodes() const
{
    QStringList res;
    switch (code_.type())
    {
    case ProductCode::Type::LabelF1:
    case ProductCode::Type::LabelF2:
    case ProductCode::Type::LabelF3:
    case ProductCode::Type::LabelF4:
    case ProductCode::Type::LabelF5:
    case ProductCode::Type::LabelF6:
        return res;
    case ProductCode::Type::LabelUnknown:
    case ProductCode::Type::LabelEan8 :
    case ProductCode::Type::LabelEan13:
    case ProductCode::Type::LabelItf14:
    case ProductCode::Type::LabelFur: res << tradeMark_.trimmed();break;
    case ProductCode::Type::LabelGs10:
    case ProductCode::Type::LabelGs1M:
    case ProductCode::Type::LabelShortCode:
    {
        res << tradeMark_
            << code_.data().mid(2, 14);
    }break;
    case ProductCode::Type::LabelEgais2:
    case ProductCode::Type::LabelEGais3:
    {
        res << tradeMark_
            << code_.data().mid(2);
    }break;
    }
//    lmWarning() << loglist(res) ;
    return res;
}

FixNumber LabelCode::getMaxPrice(bool *ok) const
{
    if(ok)*ok = false;
    FixNumber res(100u, 0ll);
    if(!code_.isGs())return res;
    if(!tradeMark_.isEmpty())
    {
        ProductCodeCreator pcc;
        LabelCode lc;
        if(pcc.checkGs1(tradeMark_, lc))
        {
            return res;
        }
        if(pcc.checkShortCode(tradeMark_, lc))
        {
            res = decodeMaxPrice(
                        lc.productCode().data().mid(
                            lc.productCode().data().size() - 4).toLatin1());
            if(ok)*ok = res.value() > 0ll;
        }
    }
    return res;
}

void LabelCode::clear()
{
    tradeMark_.clear();
    t91_.clear();
    t92_.clear();
    t93_.clear();
    code_ = ProductCode();
}

quint32 LabelCode::tailCrc() const
{
    if(t92_.isEmpty()) return 0;
    if(t92_.size() == 44)
    {
        QByteArray ba = QByteArray::fromBase64(t92_.toLatin1());
        Crc32 crc;
        return crc(ba.mid(0, 28));
    }
    return 0;
}

QVariantMap LabelCode::toMap() const
{
    return {
        {"label", tradeMark_},
        {"code", code_.toMap()},
        {"t91", t91_},
        {"t92", t92_},
        {"t93", t93_}
    };
}

QVariantMap LabelCode::toExternalMap() const
{
    return QVariantMap(); //?
}

void LabelCode::parseMap(const QVariantMap &map)
{
    clear();
    if(map.contains("label")) tradeMark_ = FormatUtils::rawLabelFromTransport(map["label"].toString().trimmed());
    else if(map.contains("rawLabel"))tradeMark_ = FormatUtils::rawLabelFromTransport(map["rawLabel"].toString().trimmed());
    ProductCodeCreator pcc;
    LabelCode lc = pcc.create(tradeMark_);
    code_ = lc.code_;
    t91_ = lc.t91_;
    t92_ = lc.t92_;
    t93_ = lc.t93_;
}

LabelCode &LabelCode::operator =(const LabelCode &other)
{
    tradeMark_ = other.tradeMark_;
    code_ = other.code_;
    t91_ = other.t91_;
    t92_ = other.t92_;
    t93_ = other.t93_;
    return  *this;
}

LabelCode &LabelCode::operator =(LabelCode &&other)
{
    tradeMark_.swap(other.tradeMark_);
    code_ = other.code_;
    t91_.swap(other.t91_);
    t92_.swap(other.t92_);
    t93_.swap(other.t93_);
    return  *this;
}

bool LabelCode::operator ==(const LabelCode &other) const
{
    return  code_ == other.code_;
}

bool LabelCode::operator !=(const LabelCode &other) const
{
    return !(*this == other);
}


LabelCode::LabelCode(const QString &tm, const ProductCode &pc)
    : tradeMark_(tm)
    , code_(pc)
    , t91_()
    , t92_()
    , t93_()
{
}

LabelCode::LabelCode(const QString &tm, const ProductCode::Type &t, const QString &code)
    : tradeMark_(tm)
    , code_(ProductCode(t, code))
    , t91_()
    , t92_()
    , t93_()
{

}

void LabelCode::setTradeMark(const QString &tradeMark)
{
    tradeMark_ = tradeMark;
}


void LabelCode::setProductCode(const ProductCode &productCode)
{
    code_ = productCode;
}


FixNumber LabelCode::decodeMaxPrice(const QByteArray &str) const
{
    FixNumber res(100u, 0ll);
    QByteArray decodeString = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!\"%&'*+-./_,:;=<>?";
    qint32 dsl = decodeString.size();
    qint32 bpl = str.size();
    qint64 buf = 0ll;
    for(int i = 0; i < bpl; ++i)
    {
        qint32 p = std::pow(dsl, bpl - i - 1);
        qint32 idx = decodeString.indexOf(str[i]);
        if(idx < 0) return res;
        buf += idx * p;
    }
    res.setValue(buf);
    return res;
}

qint64 LabelCode::intCode() const
{
    qint64 res = 0;
    if(!code_.isEmpty())
    {
        bool ok = false;
        if(!tradeMark_.isEmpty()) res = tradeMark_.toLongLong();
        else res = code_.data().toLongLong(&ok);
    }
    return res;
}



void LabelCode::setT91(const QString &newT91)
{
    t91_ = newT91;
}

void LabelCode::setT92(const QString &newT92)
{
    t92_ = newT92;
}

void LabelCode::setT93(const QString &newT93)
{
    t93_ = newT93;
}
//--------------------------------------------------------------------------------------------------
