#include "productcodecreator.h"



#include <QRegularExpression>
#include <QDate>


ProductCodeCreator::ProductCodeCreator(QObject *parent)
    : QObject(parent)
{

}

ProductCodeCreator::~ProductCodeCreator()
{

}

LabelCode ProductCodeCreator::create(const QString &barCode) const
{
    QString newLabel = barCode;
    if(newLabel.startsWith("\u001d"))newLabel = newLabel.mid(1);
    if(newLabel.isEmpty()) return LabelCode();
    LabelCode res;
    if(checkEan8(newLabel, res) ||
            checkEan13(newLabel, res) ||
            checkItf14(newLabel, res) ||
            checkGs1(newLabel, res) ||
            checkShortCode(newLabel, res) ||
            checkFur(newLabel, res) ||
            checkEgais2(newLabel, res) ||
            checkEgais3(newLabel, res))
    {
        return res;
    }
    res  = LabelCode(newLabel, ProductCode::Type::LabelUnknown, newLabel.mid(0, 32));
    return res;
}

QString ProductCodeCreator::checkAndPrepareGs1(const QString &code)
{
    QString str = code.trimmed();
    if(str.size() == 8 || str.size() == 13 || str.size() == 14 || str.size() == 29) return code;
    QRegularExpressionMatch m1, m2, m3, m4, m5;
    if(!str.contains(QRegularExpression("01(\\d{14})"), &m1) )
    {
        return code;
    }
    QString t1 = m1.capturedTexts().last();
    str = str.mid(str.indexOf(t1) + t1.length());
    QString scode = "01" +  t1;
    if(!str.contains(QRegularExpression("21([\x21-\x22\x25-\x2F\x30-\x39\x3A-\x3F\x41-\x5A\x5F\x61-\x7A]{0,20})"), &m2))
    {
        return code;
    }
    //    lmWarning() << loglist(m1.capturedTexts()) << logtab << loglist(m2.capturedTexts());


    QString t2 = m2.capturedTexts().last();
    scode += "21" + t2;
//    str = str.mid(str.indexOf(t2) + t2.length());
    QString t91, t92, t93;
    if(str.contains(QRegularExpression("91([0-9a-fA-F]{4})"), &m3))
    {
        t91 = m3.capturedTexts().last();
    }
    if(str.contains(QRegularExpression("92(?:[A-Za-z0-9+\\/]{4}\\\n?)*(?:[A-Za-z0-9+\\/]{2}==|[A-Za-z0-9+\\/]{3}=)"), &m4))
    {
        t92 = m4.capturedTexts().last().mid(2);
    }
    if(str.contains(QRegularExpression("93([\x21-\x22\x25-\x2F\x30-\x39\x3A-\x3F\x41-\x5A\x5F\x61-\x7A]{4})"), &m5))
    {
        t93 = m5.capturedTexts().last();
    }

    if((t91.isEmpty() || t92.isEmpty()) &&t93.isEmpty() )
    {
        return code;
    }
    else
    {
//        return code;
        if(code.contains('\u001d')) return code;
        str = code;
        if(str.contains("\\u001d"))str.replace("\\u001d", "\u001d");
        else if(str.contains("\r\n"))str.replace("\r\n", "\u001d");
        else if(str.contains("\n"))str.replace("\n", "\u001d");
        else if(str.contains(" "))str.replace(" ", "\u001d");
        else if(str.contains("\t"))str.replace("\t", "\u001d");
        else
        {
            if(!t91.isEmpty()) str.insert(str.lastIndexOf(t91) - 2, "\u001d");
            if(!t92.isEmpty()) str.insert(str.lastIndexOf(t92) - 2, "\u001d");
            if(!t93.isEmpty()) str.insert(str.lastIndexOf(t93) - 2, "\u001d");
        }
        return str;
    }
    return code;
}

QString ProductCodeCreator::checkAndPrepareKis(const QString &code)
{
    if(!code.contains("?kiz=")) return code;
    QString res = code.mid(code.indexOf("?kiz=") + 5).trimmed();
    LabelCode c;
    ProductCodeCreator pcc;
    return pcc.checkFur(res, c) ? res :code;
}





bool ProductCodeCreator::checkEan8(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    if(str.size() != 8 ||
            str != QStringLiteral("%1").arg(str.toUInt(), str.size(), 10, QLatin1Char('0')) ) return false;
    if(!checkBarckodeCrc(str)) return false;
    res = LabelCode(str, ProductCode::Type::LabelEan8, str);
    return true;
}

bool ProductCodeCreator::checkEan13(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    if(str.size() != 13 ||
            str != QStringLiteral("%1").arg(str.toULongLong(), str.size(), 10, QLatin1Char('0')) ) return false;
//    if(!checkBarckodeCrc(str)) return false;
    qint32 odd = 0;
    qint32 even = 0;
    for(int i = 0; i < 6; ++i)
    {
        odd += str.mid(i * 2, 1).toInt();
        even += str.mid(i * 2 + 1, 1).toInt();
    }
    even *= 3;
    qint32 num = 10 - (odd + even) % 10;
    if(num == 10) num = 0;

    if(num != str.midRef(12, 1).toInt()) return false;
    res = LabelCode(str, ProductCode::Type::LabelEan13, str);
    return true;
}

bool ProductCodeCreator::checkItf14(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    if(str.size() != 14 ||
            str != QStringLiteral("%1").arg(str.toULongLong(), str.size(), 10, QLatin1Char('0')) ) return false;
    if(!checkBarckodeCrc(str)) return false;
    res = LabelCode(str, ProductCode::Type::LabelItf14, str);
    return true;
}

bool ProductCodeCreator::checkGs1(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    if(str.size() == 8 || str.size() == 13 || str.size() == 14) return false;
    QRegularExpressionMatch m1, m2, m3, m4, m5;
    if(!str.contains(QRegularExpression("01(\\d{14})"), &m1) )
    {
        return false;
    }
    QString t1 = m1.capturedTexts().last();
    str = str.mid(str.indexOf(t1) + t1.length());
    QString scode = "01" +  t1;
    if(!str.contains(QRegularExpression("21([\x21-\x22\x25-\x2F\x30-\x39\x3A-\x3F\x41-\x5A\x5F\x61-\x7A]{0,20})"), &m2))
    {
        return false;
    }


    QString t2 = m2.capturedTexts().last();
    scode += "21" + t2;
    str = str.mid(str.indexOf(t2) + t2.length());
    QString t91, t92, t93;
    if(str.contains(QRegularExpression("91([0-9a-fA-F]{4})"), &m3))
    {
        t91 = m3.capturedTexts().last();
    }
    if(str.contains(QRegularExpression("92(?:[A-Za-z0-9+\\/]{4}\\\n?)*(?:[A-Za-z0-9+\\/]{2}==|[A-Za-z0-9+\\/]{3}=)"), &m4))
    {
        t92 = m4.capturedTexts().last().mid(2);
    }
    if(str.contains(QRegularExpression("93([\x21-\x22\x25-\x2F\x30-\x39\x3A-\x3F\x41-\x5A\x5F\x61-\x7A]{4})"), &m5))
    {
        t93 = m5.capturedTexts().last();
    }
    if((t91.isEmpty() || t92.isEmpty()) &&t93.isEmpty() )
    {
        res = LabelCode(barCode.trimmed(), ProductCode::Type::LabelGs10, scode);
    }
    else {
       res = LabelCode(barCode.trimmed(), ProductCode::Type::LabelGs1M, scode);
       res.setT91(t91);
       res.setT92(t92);
       res.setT93(t93);
    }
    return true;
}

bool ProductCodeCreator::checkShortCode(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    if(str.size() != 29) return false;
    QRegularExpressionMatch m1;
    if(!str.contains(QRegularExpression("([\x21-\x22\x25-\x2F\x30-\x39\x3A-\x3F\x41-\x5A\x5F\x61-\x7A]{0,29})"), &m1) ||
            str != m1.capturedTexts().last())
    {
        return false;
    }
    QString sn = str.mid(0, 14);
    if(sn != QStringLiteral("%1").arg(sn.toULongLong(), sn.size(), 10, QLatin1Char('0'))) return false;
    QString cs = str.mid(14, 11);

    QString scode = "01" + sn + "21" + cs;


    res = LabelCode(barCode.trimmed(), ProductCode::Type::LabelShortCode, scode);

    return true;
}

bool ProductCodeCreator::checkFur(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    QRegularExpressionMatch m;
    if(str.size() != 20 || str != str.toUpper() ||
            !str.contains(QRegularExpression("([A-Z0-9]{2}-[A-Z0-9]{6}-[A-Z0-9]{10})"), &m))
    {
        return false;
    }
    res = LabelCode(str, ProductCode::Type::LabelFur, str);
    return true;
}

bool ProductCodeCreator::checkEgais2(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    QRegularExpressionMatch m;
    if(str.size() != 68 || str != str.toUpper() || str.toUtf8() != str.toLatin1() ||
            !str.contains(QRegularExpression("([A-Z0-9]{68})"), &m))
    {
        return false;
    }
    res = LabelCode(str, ProductCode::Type::LabelEgais2, str.mid(8, 23));
    return true;
}

bool ProductCodeCreator::checkEgais3(const QString &barCode, LabelCode &res) const
{
    res.clear();
    QString str = barCode.trimmed();
    QRegularExpressionMatch m;
    if(str.size() != 150 || str != str.toUpper() || str.toUtf8() != str.toLatin1() ||
            !str.contains(QRegularExpression("([A-Z0-9]{150})"), &m))
    {
        return false;
    }
    res = LabelCode(str, ProductCode::Type::LabelEGais3, str.mid(0, 14));
    return true;
}

bool ProductCodeCreator::checkBarckodeCrc(const QString &str)
{
    LabelCode lc;
    if(str.size() == 13 && ProductCodeCreator().checkEan13(str, lc)) return true;
    qint32 odd = 0;
    qint32 even = 0;
    for(int i = 0; i < (str.size() - 1) / 2; ++i)
    {
        odd += str.mid(i * 2, 1).toInt();
        even += str.mid(i * 2 + 1, 1).toInt();
    }
    if(!(str.size() % 2)) odd += str.mid(str.size() - 2, 1).toInt();
    odd *= 3;
    qint32 num = 10 - (odd + even) % 10;
    if(num == 10) num = 0;
    return num == str.mid(str.size() - 1, 1).toInt();
}

