#include "basehttpworker.h"

#include "formatutils.h"

#include <QJsonDocument>
#include <QElapsedTimer>


const QString BaseHttpWorker::API_PREFIX = "/api012/v1/";

static const QMap<int, QString> HTTP_STATUS_TEXT = {
    {100, QStringLiteral("Continue"                        )},
    {101, QStringLiteral("Switching Protocols"             )},
    {102, QStringLiteral("Processing"                      )},
    {200, QStringLiteral("OK"                              )},
    {201, QStringLiteral("Created"                         )},
    {202, QStringLiteral("Accepted"                        )},
    {203, QStringLiteral("Non-Authoritative Information"   )},
    {204, QStringLiteral("No Content"                      )},
    {205, QStringLiteral("Reset Content"                   )},
    {206, QStringLiteral("Partial Content"                 )},
    {207, QStringLiteral("Multi-Status"                    )},
    {208, QStringLiteral("Already Reported"                )},
    {226, QStringLiteral("IM Used"                         )},
    {300, QStringLiteral("Multiple Choices"                )},
    {301, QStringLiteral("Moved Permanently"               )},
    {302, QStringLiteral("Found"                           )},
    {303, QStringLiteral("See Other"                       )},
    {304, QStringLiteral("Not Modified"                    )},
    {305, QStringLiteral("Use Proxy"                       )},
    {307, QStringLiteral("Temporary Redirect"              )},
    {308, QStringLiteral("Permanent Redirect"              )},
    {400, QStringLiteral("Bad Request"                     )},
    {401, QStringLiteral("Unauthorized"                    )},
    {402, QStringLiteral("Payment Required"                )},
    {403, QStringLiteral("Forbidden"                       )},
    {404, QStringLiteral("Not Found"                       )},
    {405, QStringLiteral("Method Not Allowed"              )},
    {406, QStringLiteral("Not Acceptable"                  )},
    {407, QStringLiteral("Proxy Authentication Required"   )},
    {408, QStringLiteral("Request Timeout"                 )},
    {409, QStringLiteral("Conflict"                        )},
    {410, QStringLiteral("Gone"                            )},
    {411, QStringLiteral("Length Required"                 )},
    {412, QStringLiteral("Precondition Failed"             )},
    {413, QStringLiteral("Payload Too Large"               )},
    {414, QStringLiteral("URI Too Long"                    )},
    {415, QStringLiteral("Unsupported Media Type"          )},
    {416, QStringLiteral("Range Not Satisfiable"           )},
    {417, QStringLiteral("Expectation Failed"              )},
    {421, QStringLiteral("Misdirected Request"             )},
    {422, QStringLiteral("Unprocessable Entity"            )},
    {423, QStringLiteral("Locked"                          )},
    {424, QStringLiteral("Failed Dependency"               )},
    {426, QStringLiteral("Upgrade Required"                )},
    {428, QStringLiteral("Precondition Required"           )},
    {429, QStringLiteral("Too Many Requests"               )},
    {431, QStringLiteral("Request Header Fields Too Large" )},
    {451, QStringLiteral("Unavailable For Legal Reasons"   )},
    {500, QStringLiteral("Internal Server Error"           )},
    {501, QStringLiteral("Not Implemented"                 )},
    {502, QStringLiteral("Bad Gateway"                     )},
    {503, QStringLiteral("Service Unavailable"             )},
    {504, QStringLiteral("Gateway Timeout"                 )},
    {505, QStringLiteral("HTTP Version Not Supported"      )},
    {506, QStringLiteral("Variant Also Negotiates"         )},
    {507, QStringLiteral("Insufficient Storage"            )},
    {508, QStringLiteral("Loop Detected"                   )},
    {510, QStringLiteral("Not Extended"                    )},
    {511, QStringLiteral("Network Authentication Required" )},
};

BaseHttpWorker::BaseHttpWorker(QObject *parent)
    : QObject(parent)
    , api_(new HttpCashboxServiceApi(this))
    , cashier_()
{

}

BaseHttpWorker::~BaseHttpWorker()
{

}

QHttpServerResponse BaseHttpWorker::operator ()(const QHttpServerRequest &req)
{
    QElapsedTimer timer;
    timer.start();
    cashier_ = Cashier();
    if(!api_ || !api_->isReady())
    {
        return QHttpServerResponse(
                    "application/json",
                    getJsonResultAnswer(QHttpServerResponse::StatusCode::BadGateway),
                    QHttpServerResponse::StatusCode::BadGateway);
    }
    if(req.method() == QHttpServerRequest::Method::OPTIONS)
    {
        return sendOptions();
    }
    if(needAutorizing(req))
    {
        CoreApiResult car = login(req);
        if(!car.isOk())
        {

            QHttpServerResponse response(
                        "application/json",
                        getJsonResultAnswer(QHttpServerResponse::StatusCode::Unauthorized,
                                            car.descr()),
                        QHttpServerResponse::StatusCode::Unauthorized);
            response.setHeader("WWW-Authenticate", "Basic realm=\"nmrs_m7VKmomQ2YM3:\"");
            return response;

        }
    }



    QHttpServerResponse resp =  exec(req);
    if(resp.statusCode() != QHttpServerResponse::StatusCode::Ok && resp.data().isEmpty())
    {
        resp = QHttpServerResponse("application/json",
                                   getJsonResultAnswer(resp.statusCode()),
                                   resp.statusCode());
    }

    return resp;
}

QHttpServerResponse BaseHttpWorker::sendOptions() const
{
    QHttpServerResponse resp(QHttpServerResponse::StatusCode::Ok);
    setHeaders(resp);
    return resp;
}

void BaseHttpWorker::setHeaders(QHttpServerResponse &resp) const
{
    //TODO: add origins
//    QStringList origins = cfg.allowedOrigins();
//	for(const QString &s :origins) response.setHeader("Access-Control-Allow-Origin", s.toUtf8());
    resp.addHeader("Access-Control-Allow-Credentials", "true");
    resp.addHeader("Access-Control-Allow-Methods", "GET, POST, OPTIONS");
    resp.addHeader("Access-Control-Allow-Headers",
                       "DNT,X-CustomHeader,Keep-Alive,User-Agent,X-Requested-With,"
                       "If-Modified-Since,Cache-Control,Content-Type,"
                       "Content-Range,Range,Authorization");
}

CoreApiResult BaseHttpWorker::login(const QHttpServerRequest &req)
{
    QString user;
    QString password;
    QByteArray auth = req.headers().value("Authorization").toString().toUtf8();
    if((auth.isEmpty() || !auth.startsWith("Basic ")))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::AccessDenied);
    }
    else
    {
        auth = QByteArray::fromBase64(auth.mid(6).trimmed());
        QString sauth = QString::fromUtf8(auth);
        user = sauth.mid(0, sauth.indexOf(":"));
        password = sauth.mid(sauth.indexOf(":") + 1);
    }
    QMap<QString, QString> params;
    CoreTransaction transaction = prepareTransaction(req, params);

    transaction.setOperation(CoreTransaction::Operation::LOGIN);
    transaction.genUid();
    transaction.setCashierLogin(user);
    transaction.setCashierPassword(password);
    transaction.setFormatToExternal(false);
//    lmWarning() << logvariant(transaction.toMap());
    CoreApiResult res = api_->exec(transaction);
//    lmWarning() << logvariant(res.toMap());
    if(res.isOk())
    {
        cashier_.setMap(res.data());
    }
    return res;
}

bool BaseHttpWorker::needAutorizing(const QHttpServerRequest &req) const
{
    Q_UNUSED(req)
    return true;
}

QString BaseHttpWorker::getHttpResultDescription(int status) const
{
    switch (status)
    {
    case 100: return QStringLiteral("Continue");
    case 101: return QStringLiteral("Switching Protocols");
    case 102: return QStringLiteral("Processing");
    case 103: return QStringLiteral("Early Hints");
    case 200: return QStringLiteral("OK");
    case 201: return QStringLiteral("Created");
    case 202: return QStringLiteral("Accepted");
    case 203: return QStringLiteral("Non-Authoritative Information");
    case 204: return QStringLiteral("No Content");
    case 205: return QStringLiteral("Reset Content");
    case 206: return QStringLiteral("Partial Content");
    case 207: return QStringLiteral("Multi-Status");
    case 208: return QStringLiteral("Already Reported");
    case 226: return QStringLiteral("IM Used");
    case 300: return QStringLiteral("Multiple Choices");
    case 301: return QStringLiteral("Moved Permanently");
    case 302: return QStringLiteral("Moved Temporarily");
    case 303: return QStringLiteral("See Other");
    case 304: return QStringLiteral("Not Modified");
    case 305: return QStringLiteral("Use Proxy");
    case 307: return QStringLiteral("Temporary Redirect");
    case 308: return QStringLiteral("Permanent Redirect");
    case 400: return QStringLiteral("Bad Request");
    case 401: return QStringLiteral("Unauthorized");
    case 402: return QStringLiteral("Payment Required");
    case 403: return QStringLiteral("Forbidden");
    case 404: return QStringLiteral("Not Found");
    case 405: return QStringLiteral("Method Not Allowed");
    case 406: return QStringLiteral("Not Acceptable");
    case 407: return QStringLiteral("Proxy Authentication Required");
    case 408: return QStringLiteral("Request Timeout");
    case 409: return QStringLiteral("Conflict");
    case 410: return QStringLiteral("Gone");
    case 411: return QStringLiteral("Length Required");
    case 412: return QStringLiteral("Precondition Failed");
    case 413: return QStringLiteral("Payload Too Large");
    case 414: return QStringLiteral("URI Too Long");
    case 415: return QStringLiteral("Unsupported Media Type");
    case 416: return QStringLiteral("Range Not Satisfiable");
    case 417: return QStringLiteral("Expectation Failed");
    case 418: return QStringLiteral("I’m a teapot");
    case 419: return QStringLiteral("Authentication Timeout");
    case 421: return QStringLiteral("Misdirected Request");
    case 422: return QStringLiteral("Unprocessable Entity");
    case 423: return QStringLiteral("Locked");
    case 424: return QStringLiteral("Failed Dependency");
    case 425: return QStringLiteral("Too Early");
    case 426: return QStringLiteral("Upgrade Required");
    case 428: return QStringLiteral("Precondition Required");
    case 429: return QStringLiteral("Too Many Requests");
    case 431: return QStringLiteral("Request Header Fields Too Large");
    case 449: return QStringLiteral("Retry With");
    case 451: return QStringLiteral("Unavailable For Legal Reasons");
    case 499: return QStringLiteral("Client Closed Request");
    case 500: return QStringLiteral("Internal Server Error");
    case 501: return QStringLiteral("Not Implemented");
    case 502: return QStringLiteral("Bad Gateway");
    case 503: return QStringLiteral("Service Unavailable");
    case 504: return QStringLiteral("Gateway Timeout");
    case 505: return QStringLiteral("HTTP Version Not Supported");
    case 506: return QStringLiteral("Variant Also Negotiates");
    case 507: return QStringLiteral("Insufficient Storage");
    case 508: return QStringLiteral("Loop Detected");
    case 509: return QStringLiteral("Bandwidth Limit Exceeded");
    case 510: return QStringLiteral("Not Extended");
    case 511: return QStringLiteral("Network Authentication Required");
    case 520: return QStringLiteral("Unknown Error");
    case 521: return QStringLiteral("Web Server Is Down");
    case 522: return QStringLiteral("Connection Timed Out");
    case 523: return QStringLiteral("Origin Is Unreachable");
    case 524: return QStringLiteral("A Timeout Occurred");
    case 525: return QStringLiteral("SSL Handshake Failed");
    case 526: return QStringLiteral("Invalid SSL Certificate");
    }
    return QStringLiteral("Unhandled status: %1").arg(status);
}

QString BaseHttpWorker::getHtmlResultAnswer(int status, const QString &text) const
{
    if(status == 200) return QString();
    QString res =
            "<!DOCTYPE html>"
            "<html>"
            "    <head>"
            "        <meta charset=\"utf-8\">"
            "        <title>HTTP %1</title>"
            "    </head>"
            "    <body>"
            "        <p>%2</p>"
            "    </body>"
            "</html>";
    QString txt = text.isEmpty() ? getHttpResultDescription(status) : text;
    txt = QString::fromLatin1(transliterate(txt).toLatin1());
    res = res.arg(status).arg(txt);
    return res;
}

QString BaseHttpWorker::getHtmlResultAnswer(QHttpServerResponse::StatusCode status, const QString &text) const
{
    return getHtmlResultAnswer(static_cast<qint32>(status), text);
}

QByteArray BaseHttpWorker::getJsonResultAnswer(int status, const QString &text) const
{
    QString txt = text;
    if(txt.isEmpty() && HTTP_STATUS_TEXT.contains(status))
    {
        txt = HTTP_STATUS_TEXT[status];
    }
    QVariantMap res = {
        {"result", static_cast<qint32>(CoreApiConst::ErrorCode::BaseHttpResult) | status},
        {"message", QString::fromLatin1(transliterate(txt).toLatin1())}
    };
    return QJsonDocument::fromVariant(res).toJson();
}

QByteArray BaseHttpWorker::getJsonResultAnswer(QHttpServerResponse::StatusCode status, const QString &text) const
{
    return getJsonResultAnswer(static_cast<qint32>(status), text);
}

QHttpServerResponse BaseHttpWorker::documentToResponse(const CoreApiResult &res) const
{
    QVariantMap data;
    if(res.isOk())
    {
        data.insert("result", 0);
        data.insert("document", res.data());
    }
    else
    {
        data.insert("result", static_cast<qint32>(res.code()));
        data.insert("message", res.descr());
        if(!res.data().isEmpty()) data.insert("document", res.data());
    }
    QJsonDocument jdoc = QJsonDocument::fromVariant(data);
//    QHttpServerResponse resp(jdoc.object());
    QHttpServerResponse resp("application/json", jdoc.toJson());
    setHeaders(resp);
    return resp;
}

QHttpServerResponse BaseHttpWorker::dataToResponse(const CoreApiResult &res, bool raw, const QString &name) const
{
    QVariantMap data;
    if(raw) data = res.toMap();
    else if(res.isOk())
    {
        data.insert("result", 0);
        if(name.isEmpty()) data = UNIT_MAPS(data, res.data());
        else data.insert(name, res.data());
    }
    else
    {
        data.insert("result", static_cast<qint32>(res.code()));
        data.insert("message", res.descr());
        if(!res.data().isEmpty()) data.insert(name, res.data());
    }
//    lmWarning() << logvariant(data) << logtab << logvariant(res.toMap());
    QJsonDocument jdoc = QJsonDocument::fromVariant(data);
//    QHttpServerResponse resp(jdoc.object());
    QHttpServerResponse resp("application/json", jdoc.toJson());
    setHeaders(resp);
    return resp;
}

QMap<QString, QString> BaseHttpWorker::getQueryItems(const QHttpServerRequest &req) const
{
    QUrlQuery uq = req.query();
    QList<QPair<QString, QString> > items = uq.queryItems();
    QMap<QString, QString> res;
    for(const QPair<QString, QString>&i:items)
    {
        res.insert(i.first.toLower(), i.second);
    }
    return res;
}

CoreTransaction BaseHttpWorker::prepareTransaction(const QHttpServerRequest &req, QMap<QString, QString> &params) const
{
//    lmWarning() << req.url().path() << logvariant(req.headers());
    CoreTransaction trans;
    trans.setFormatToExternal(true);
    params = getQueryItems(req);
    if(params.contains("externalid") && !params["externalid"].trimmed().isEmpty())
    {
        trans.setExternalId(params["externalid"].trimmed());
    }
    if(params.contains("clientid") && !params["clientid"].trimmed().isEmpty())
    {
        trans.setClientId(params["clientid"].trimmed());
    }
    if(params.contains("raw") && params["raw"].toInt())
    {
        trans.setFormatToExternal(false);
    }
    if(params.contains("cdt"))
    {
        QString cdt = QString::fromUtf8(QByteArray::fromBase64(params["cdt"].toLatin1())).trimmed();
        trans.setClientDateTime(STR2DT_(cdt));
    }
    return trans;
}

QString BaseHttpWorker::getPath(const QString &name)
{
    return API_PREFIX + name;
}

QVariantMap BaseHttpWorker::lowHeaders(const QHttpServerRequest &req) const
{
    QVariantMap h = req.headers();
    QVariantMap res;
    for(auto it = h.constBegin(); it != h.constEnd(); ++it)
    {
        res.insert(it.key().toLower(), it.value());
    }
    return res;
}

QHttpServerResponse BaseHttpWorker::uploadFile(const QString &rootFolder, const QString &pathPrefix,
                                               qint32 maxSize, const QHttpServerRequest &req)
{
    Q_UNUSED(pathPrefix)
    QString fn = req.url().path().mid(req.url().path().lastIndexOf("/")).trimmed();
    fn = rootFolder + fn;
    QFileInfo fi(fn);
    QVariantMap hs = lowHeaders(req);
    if(fi.isDir() || !hs.contains("content-length") ||
        hs.value("content-length").toInt() <= 0 ||
        hs.value("content-length").toInt() > maxSize)
    {
        return QHttpServerResponse::StatusCode::BadRequest;
    }

    QFile f(fn);
    if(!f.open(QIODevice::WriteOnly))
    {
        return QHttpServerResponse::StatusCode::Forbidden;
    }
    f.write(req.body());
    f.close();
    return QHttpServerResponse::StatusCode::Ok;
}

QHttpServerResponse BaseHttpWorker::downloadFile(const QString &rootFolder, const QString &pathPrefix, const QHttpServerRequest &req)
{
    QString fn = req.url().path().mid(req.url().path().lastIndexOf("/")).trimmed().remove("/");
    if(fn == pathPrefix) fn = "/";
    else fn = "/" + fn;
    fn = rootFolder + fn;
    QFileInfo fi(fn);
    if(fi.isDir())
    {
        return folderToHtml(fi.absoluteFilePath(), req.url().toString());
    }
    if(fi.fileName().trimmed().toLower() == "list.json")
    {
        return folderToJson(fi.absolutePath(), req.url().toString());
    }
    return QHttpServerResponse::fromFile(fn);
}

QHttpServerResponse BaseHttpWorker::folderToHtml(const QString &folder, const QString &urlPrefix, bool images) const
{
    CoreApiResult result = api_->loadCashboxStatus(true);
    CashboxStatus st(result.data());
    QString u = urlPrefix;
    if(!u.endsWith("/")) u += "/";
    QDir d(folder);
    QStringList sl = d.entryList(QDir::Files);
    QStringList res;
    res << "<html><head>"
        << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">"
        << "</head><body>" ;
    res << st.modelData().toHtml();
    res << QStringLiteral("<h3 align=\"center\"> %1*</h3>").arg(u);
    for(const QString &s: sl)
    {
        if(images)
        {
            res << QStringLiteral("<p><img src=\"%1%2\" width=\"64\" height=\"64\">  <a href=\"%1%2\">%2</a></p>")
                       .arg(u).arg(s);
        }
        else
        {
            res << QStringLiteral("<p><a href=\"%1%2\">%2</a></p>").arg(u).arg(s);
        }
    }
    res << "</body></html>";
    QHttpServerResponse resp("text/html", res.join("\n").toUtf8());
    setHeaders(resp);
    return resp;
}

QHttpServerResponse BaseHttpWorker::folderToJson(const QString &folder, const QString &urlPrefix) const
{
    QString u = urlPrefix;
    if(!u.endsWith("/")) u += "/";

    QDir d(folder);
    QFileInfoList sl = d.entryInfoList(QDir::Files);
    QVariantList res;
    for(const QFileInfo &fi: sl)
    {
        QVariantMap m = {
            {"name", fi.fileName()},
            {"url", u + fi.fileName()},
            {"size", fi.size()}
        };
        res << m;
    }
    QHttpServerResponse resp("application/json", QJsonDocument::fromVariant(res).toJson());
    setHeaders(resp);
    return resp;
}

QString BaseHttpWorker::transliterate(const QString &src) const
{
    static const QString rusUpper = QString::fromUtf8("АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ№");
    static const QStringList TRNS{
        "A", "B", "V", "G", "D", "E", "JO", "ZH", "Z", "I", "J", "K",
        "L", "M", "N", "O", "P", "R", "S", "T", "U", "F", "H", "TS", "CH",
        "SH", "SCH", "'", "Y", "'", "E", "JU", "JA", "#"
    };
    QString res;
    for(int i = 0; i < src.size(); ++i)
    {
        qint32 idx = rusUpper.indexOf(src.toUpper().at(i));
        if(idx >= 0 && idx < TRNS.size())
        {
            res = res + TRNS[idx];
        }
        else
        {
            res = res + src[i];
        }
    }
    return res;
}

