#include "reportshttpworker.h"

#include "appdirectories.h"
#include "zip.h"

ReportsHttpWorker::ReportsHttpWorker(QObject *parent)
    : BaseHttpWorker(parent)
{

}

ReportsHttpWorker::~ReportsHttpWorker()
{

}

void ReportsHttpWorker::initRoutes(QHttpServer *server)
{
    server->route(getPath("findfsdoc.json"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
    server->route(getPath("ofdticket.json"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
    server->route(getPath("fiscaldoc.json"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
    server->route(getPath("tlvdoc.json"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
    server->route(getPath("regdoc.json"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
    server->route(getPath("offlineNotifications.bin"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
    server->route(getPath("offlineNotifications.zip"), [](const QHttpServerRequest &req) {
        ReportsHttpWorker w;
        return w(req);
    });
}

QHttpServerResponse ReportsHttpWorker::exec(const QHttpServerRequest &req)
{
    if(req.url().path().toLower().startsWith(getPath("offlinenotifications")))
    {
        return offlineNotifications(req);
    }
    if(req.method() == QHttpServerRequest::Method::POST) return execPost(req);
    if(req.method() == QHttpServerRequest::Method::GET) return execGet(req);
    return QHttpServerResponse::StatusCode::NotFound;
}

QHttpServerResponse ReportsHttpWorker::execPost(const QHttpServerRequest &req)
{
    QHttpServerResponse res(QHttpServerResponse::StatusCode::NotFound);
    CoreTransaction trans;
    if(!postReqToTransaction(req, trans, res)) return res;

    CoreApiResult result = api_->exec(trans);
    return documentToResponse(result);
}

QHttpServerResponse ReportsHttpWorker::execGet(const QHttpServerRequest &req)
{
    QHttpServerResponse res(QHttpServerResponse::StatusCode::NotFound);
    CoreTransaction trans;
    if(!getReqToTransaction(req, trans, res)) return res;

    CoreApiResult result = api_->exec(trans);
//    lmWarning() << logvariant(result.toMap());
    return documentToResponse(result);
}

bool ReportsHttpWorker::getReqToTransaction(const QHttpServerRequest &req,
                                            CoreTransaction &trans,
                                            QHttpServerResponse &resp)
{
    QMap<QString, QString> params;
    trans = prepareTransaction(req, params);
    QVariantMap data;
    if(req.url().path().toLower().startsWith(getPath("findfsdoc")))
    {
        trans.setOperation(CoreTransaction::Operation::FIND_FS_DOC);
        if(params.contains("fd")) data["fd"] = params["fd"].toInt();
        else if(params.contains("number")) data["fd"] = params["number"].toInt();
    }
    else if(req.url().path().toLower().startsWith(getPath("ofdticket")))
    {
        trans.setOperation(CoreTransaction::Operation::FIND_OFD_TICKET);
        if(params.contains("fd")) data["fd"] = params["fd"].toInt();
        else if(params.contains("number")) data["fd"] = params["number"].toInt();
    }
    else if(req.url().path().toLower().startsWith(getPath("fiscaldoc")) ||
            req.url().path().toLower().startsWith(getPath("tlvdoc")))
    {
        trans.setOperation(CoreTransaction::Operation::GET_TLV_DOC);
        if(params.contains("fd")) data["fd"] = params["fd"].toInt();
        else if(params.contains("number")) data["fd"] = params["number"].toInt();
    }
    else if(req.url().path().toLower().startsWith(getPath("regdoc")) )
    {
        bool sh = false;
        if(params.contains("short")&& !params["short"].trimmed().isEmpty())
        {
            bool ok = false;
            QString s = params["short"].trimmed();
            sh = (s.toInt(&ok) && ok) || s.toLower() == "true";
        }
        trans.setOperation(sh ? CoreTransaction::Operation::GET_REG_ARCHIVE_DOC :
                                CoreTransaction::Operation::GET_REG_TLV_DOC);
        if(params.contains("number")) data["number"] = params["number"].toInt();
    }
    else
    {
        resp = QHttpServerResponse(QHttpServerResponse::StatusCode::NotFound);
        return false;
    }
    trans.setParams(data);
    trans.setCashierLogin(cashier_.login());
    trans.setCashierPassword(cashier_.password());
    trans.setUid(trans.genUid());
    return true;
}

bool ReportsHttpWorker::postReqToTransaction(const QHttpServerRequest &req, CoreTransaction &trans, QHttpServerResponse &resp)
{
    resp = QHttpServerResponse(QHttpServerResponse::StatusCode::NotFound);
    return false;
}

QHttpServerResponse ReportsHttpWorker::offlineNotifications(const QHttpServerRequest &req)
{
    if(!req.url().path().toLower().startsWith(getPath("offlinenotifications")))
    {
        return QHttpServerResponse::StatusCode::NotFound;
    }
    CoreTransaction trans;

    QMap<QString, QString> params;

    QVariantMap data;
    trans = prepareTransaction(req, params);


    QString fn = APATH.offlineNotifications() + "/" + trans.externalId();


    auto prepareFile = [&req](const QString &fileName) -> QHttpServerResponse {
        if(req.url().path().toLower().endsWith(".zip"))
        {
            if(!QFile::exists(fileName + ".zip"))
            {
                Zip zip;
                zip.createArchive(fileName + ".zip");
                zip.addFile(fileName, Zip::Deflate9);
            }
            return QHttpServerResponse::fromFile(fileName + ".zip");
        }
        else
        {
            return QHttpServerResponse::fromFile(fileName);
        }
    };

    if(QFile::exists(fn)) return prepareFile(fn);


    trans.setParams(data);
    trans.setCashierLogin(cashier_.login());
    trans.setCashierPassword(cashier_.password());
    trans.setOperation(CoreApiConst::RemoteCommand::GET_OFFLINE_NOTIFICATIONS);
    trans.setUid(trans.genUid());

    CoreApiResult result = api_->exec(trans);
//    lmWarning() << logvariant(result.toMap());
    if(!result.isOk())
    {
        if(result.code() == CoreApiConst::ErrorCode::NoDataRequested ||
                result.code() == CoreApiConst::ErrorCode::OfflineModeOnly)
        {
            return QHttpServerResponse::StatusCode::NoContent;
        }
        return QHttpServerResponse::StatusCode::InternalServerError;
    }
    fn = result.data()["fileName"].toString().trimmed();
    if(!QFile::exists(fn))
    {
        return QHttpServerResponse::StatusCode::InternalServerError;
    }
    return prepareFile(fn);
}
