#include "fswrapper.h"

#include "fsapi.h"

#include "regdoc.h"
#include "formatutils.h"
#include "internalobjectsorage.h"
#include "coreruntimestorage.h"


#include <QElapsedTimer>
#include <QDataStream>
#include <QTextCodec>

static const QList<quint16> REG_DOC_PARAMS = {
    1001, 1002, 1009, 1012, 1013, 1017, 1018, 1021, 1036, 1037,
    1040, 1041, 1046, 1048, 1056, 1060, 1062, 1077, 1117, 1187,
    1188, 1189, 1190, 1203, 1209, 1213, 1274, 1275, 1290, 1205, 1157
};


FsWrapper::FsWrapper(QObject *parent)
    : QObject(parent)
    , api_(new FsApi(this))
{

}

FsWrapper::~FsWrapper()
{

}

bool FsWrapper::open()
{
    return api_ && api_->open();
}

void FsWrapper::close()
{
    if(api_)api_->close();
}

bool FsWrapper::isOpen() const
{
    return api_ && api_->isOpen();
}

CoreApiConst::ErrorCode FsWrapper::getFullFsStatus(FsFullStatus &status,
                                                   const fdf::RegFlags &flags,
                                                   const fdf::RegFlags1290 &extFlags)
{
    QElapsedTimer t;
    t.start();
    CoreApiConst::ErrorCode res = CoreApiConst::ErrorCode::Ok;
    if(status.fsNumber().isEmpty() || status.hasError())
    {
        QString serial;
        res = getFsNumber(serial);
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setFsNumber(serial);
    }
    if(status.supportedFfd() == fs::FFD::Invalid|| status.hasError())
    {
        fs::FFD cf, sf;
        res = getFsFormat(cf, sf);
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setCurrentFfd(cf);
        status.setSupportedFfd(sf);
    }
    if(!status.hasStatus()|| status.hasError())
    {
        std::optional<FsStatus> fsstatus{};
        res = getFsStatus(fsstatus);
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setStatus(fsstatus);
    }
    bool registered = status.hasStatus() && status.status().phase() > fs::Phase::Empty;
    if(!status.hasLifeTime()|| status.hasError())
    {
        std::optional<FsLifeTime> lt{};
        res = getFsLifeTime(lt);
//        lmWarning() << static_cast<qint32>(res) << lt.has_value();
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setLifeTime(lt);
    }
    if(registered && ((status.hasLifeTime() && status.lifeTime().expiredDays() < 0)|| status.hasError()))
    {
        qint32 days = -1;
        res = getFsLifedays(days);
        FsLifeTime lt = status.lifeTime();
        lt.setExpiredDays(days);
        status.setLifeTime(lt);
    }
    if(status.version().isEmpty()|| status.hasError())
    {
        QString version;
        bool isRelease = true;
        res = getFsVersion(version, isRelease);
//        lmWarning() << static_cast<qint32>(res) << version << isRelease;
        if(res != CoreApiConst::ErrorCode::Ok) return res;
        status.setVersion(version);
        status.setIsRelease(isRelease);
    }
    if(status.release().isEmpty()|| status.hasError())
    {
        QString release;
        res = getFsRelease(release);
//        lmWarning() << static_cast<qint32>(res) << release;
        if(res != CoreApiConst::ErrorCode::Ok) return res;
        status.setRelease(release);
    }
    if(!status.hasMemResource()|| status.hasError())
    {
        std::optional<FsMemResourceInfo> resource{};
        res = getFMemResource(resource);
//        lmWarning() << static_cast<qint32>(res) << resource.has_value();
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setMemResource(resource);
    }
//    lmWarning() << "LOAD_CYCLE_STATUS";
    if(!status.hasCycle()|| status.hasError())
    {
        std::optional<CycleStatus> c{};
        res = getFsCycle(c);
//        lmWarning() << static_cast<qint32>(res) << c.has_value();
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setCycle(c);
        if(c.has_value() && !status.cycleOpenDt().isValid())
        {
            QDateTime o, cl;
            res = getCycleDates(status, o, cl);
            if(res == CoreApiConst::ErrorCode::Ok)
            {
                status.setCycleOpenDt(o);
                status.setCycleCloseDt(cl);

            }
        }

    }
//    lmWarning() << "LOAD_TRANSPORT_STATUS";
    if(status.isFiscalized() && !status.hasTranspot())
    {
        std::optional<FsTransportState> ts{};
        res = getFsTransport(ts);
//        lmWarning() << static_cast<qint32>(res) << ts.has_value();
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setTransport(ts);
    }

    if(status.cashboxSerial().isEmpty() && status.isFiscalized())
    {
        Tlv tlv;
        res = readRegTlv(status.lifeTime().regsCount(), fdf::Tag::CashboxSerial, tlv);
//        lmWarning() << static_cast<qint32>(res) << tlv.toString();
        if(res != CoreApiConst::ErrorCode::Ok)
        {
            status.setError(res);
            return res;
        }
        status.setCashboxSerial(tlv.toString().trimmed());
    }
    if(!status.hasLabelsTranspot() && status.isFiscalized() &&
            !flags.testFlag(fdf::RegFlag::OfflineMode) &&
            extFlags.testFlag(fdf::RegFlag1290::LabledProducts))
    {
        RegData rd = IOSG::getRegData();
        if(!rd.getOfflineMode() && rd.extFlags().getLabledProducts())
        {
            std::optional<LabelsTransportState> st;
            res = readLabelsTransport(st);
            if(res != CoreApiConst::ErrorCode::Ok)
            {
                status.setError(res);
                return res;
            }
            status.setLabelsTransport(st);
        }
    }
    status.setError(CoreApiConst::ErrorCode::Ok);
//    lmWarning() << "FULL STATUS IS READY FOR " << t.elapsed() << logvariant(status.toMap());
    return res;
}

CoreApiConst::ErrorCode FsWrapper::getFsStatus(std::optional<FsStatus> &status)
{
    status.reset();
    FsReq req(fs::ReqCode::FsState);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return parsingResultToError(rep.getFsStatus(status));

}

CoreApiConst::ErrorCode FsWrapper::getFsNumber(QString &serial)
{
    serial.clear();
    FsReq req(fs::ReqCode::FsNumber);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    return parsingResultToError(rep.getFsNumber(serial));

}

CoreApiConst::ErrorCode FsWrapper::getFsLifeTime(std::optional<FsLifeTime> &lt)
{
    lt.reset();
    FsReq req(fs::ReqCode::FsLifetime);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return parsingResultToError(rep.getFsLifeTime(lt));
}

CoreApiConst::ErrorCode FsWrapper::getFsVersion(QString &version, bool &isRelease)
{
    version.clear();
    isRelease = true;
    FsReq req(fs::ReqCode::FsVersion);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    return parsingResultToError(rep.getFsVersion(version, isRelease));
}

CoreApiConst::ErrorCode FsWrapper::getFsLastErrors(QByteArray &dump)
{
    dump.clear();
    FsReq req(fs::ReqCode::FsErrors);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    return parsingResultToError(rep.getLastFsErrors(dump));
}

CoreApiConst::ErrorCode FsWrapper::getFsRelease(QString &release)
{
    release.clear();
    FsReq req(fs::ReqCode::FsRelease);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    return parsingResultToError(rep.getFsRelease(release));
}

CoreApiConst::ErrorCode FsWrapper::getFsFormat(fs::FFD &curFfd, fs::FFD &supportedFfd)
{
    curFfd = fs::FFD::Invalid;
    supportedFfd = fs::FFD::Invalid;
    FsReq req(fs::ReqCode::FsFormat);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    return parsingResultToError(rep.getFsFormat(curFfd, supportedFfd));
}

CoreApiConst::ErrorCode FsWrapper::getFsLifedays(qint32 &days, const QDateTime &dt)
{
    days = -1;
    FsReq req(fs::ReqCode::FsLifeDays, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    return parsingResultToError(rep.getExpiredDays(days));
}

CoreApiConst::ErrorCode FsWrapper::getFMemResource(std::optional<FsMemResourceInfo> &resource)
{
    resource.reset();
    FsReq req(fs::ReqCode::FsFreeMem);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return parsingResultToError(rep.getMemResource(resource));
}

CoreApiConst::ErrorCode FsWrapper::getFsCycle(std::optional<CycleStatus> &cycle)
{
    cycle.reset();
    FsReq req(fs::ReqCode::CurrentCycleData);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return parsingResultToError(rep.getFsCycle(cycle));
}

CoreApiConst::ErrorCode FsWrapper::getFsTransport(std::optional<FsTransportState> &st)
{
    st.reset();
    FsReq req(fs::ReqCode::TransfereStatus);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(8000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return parsingResultToError(rep.getFsTransport(st));
}

CoreApiConst::ErrorCode FsWrapper::cleanDebugFs()
{
    FsReq req(fs::ReqCode::ResetFsState, QByteArray(1, '\x16'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(60000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::cancelDoc()
{
    FsReq req(fs::ReqCode::CancelDocument);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return rep.repError();
}

CoreApiConst::ErrorCode FsWrapper::sendDocData(const QByteArray &data)
{
    FsReq req(fs::ReqCode::SendDocumentData, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(60000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return rep.repError();
}

CoreApiConst::ErrorCode FsWrapper::sendDocData(const Tlv::Stlv &list)
{
    if(list.isEmpty()) return CoreApiConst::ErrorCode::Ok;
    QByteArray buf;
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    for(const Tlv &t: list)
    {
        if(buf.size() + t.rawLen() > 1024)
        {
            err = sendDocData(buf);
            if(err != CoreApiConst::ErrorCode::Ok) return err;
            buf = t.rawData();
        }
        else buf.append(t.rawData());
    }
    if(!buf.isEmpty()) return sendDocData(buf);
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readRegTlv(quint8 seq, fdf::Tag tag, Tlv &val)
{
    val.clean();
    QByteArray  buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    if(seq)
    {
        ds << seq << quint16(tag);
    }
    else
    {
        ds << quint16(tag);
    }
    FsReq req(fs::ReqCode::FiscalisationParameter, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(60000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    if(seq) val.setRawData(rep.data());
    else
    {
        val.setTag(tag);
        val.setRawData(rep.data());
    }
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readTlvDoc(quint32 docNumber, quint16 &type, Tlv::Stlv &list)
{
    list.clear();
    type = 0u;

    FsReq req;
    FsRep rep;
    {
        QByteArray data;
        QDataStream ds(&data, QIODevice::WriteOnly);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds << docNumber;
        req = FsReq(fs::ReqCode::FsDocumentTLV , data);
    }
    CoreApiConst::ErrorCode  err = exec(req, rep, qMakePair(20000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return err;
    }
    {
        quint16 docLen = 0u;
        QByteArray buf = rep.data();
        QDataStream ds(buf);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds >> type >> docLen;
    }
    req = FsReq(fs::ReqCode::FsDocumentTLVReading);
    do
    {
        exec(req, rep, qMakePair(10000, 1000));
        if (rep.repCode() == fs::RepCode::Ok || (rep.repCode() == fs::RepCode::NoDataRequested &&
                                                                rep.data().size() > 2))
        {

            QDataStream ds(rep.data());
            ds.setByteOrder(QDataStream::LittleEndian);
            while(!ds.atEnd())
            {
                Tlv t;
                ds >> t;
                if(t.isValid()) list << t;
            }
        }
    }
    while(rep.repCode() == fs::RepCode::Ok);
    if(rep.repCode() != fs::RepCode::Ok &&
            rep.repCode() != fs::RepCode::NoDataRequested)
    {
        type = 0u;
        list.clear();
        return rep.repError();
    }
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readRegData(quint8 seq, RegData &rd)
{
    Tlv::Stlv list;
    for(quint16 t : REG_DOC_PARAMS)
    {
        Tlv tlv;
        CoreApiConst::ErrorCode err = readRegTlv(seq, static_cast<fdf::Tag>(t), tlv);
        if(err == CoreApiConst::ErrorCode::Ok)
        {
            list << tlv;
        }
        else
        {
            if(err != CoreApiConst::ErrorCode::NoDataRequested) return err;
        }
    }
    RegDoc doc;
    doc.parseFromTlvList(list);
    rd = doc.regData();
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readRegDoc(quint8 seq, BaseRegDoc *rd)
{
    rd = nullptr;
    PFsArchiveReportBase regrep;
    CoreApiConst::ErrorCode err = findRegDoc(seq, regrep, QStringList());
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(!regrep) return CoreApiConst::ErrorCode::UnknownFsError;
    Tlv::Stlv list;
    for(quint16 t : REG_DOC_PARAMS)
    {
        Tlv tlv;
        err = readRegTlv(seq, static_cast<fdf::Tag>(t), tlv);
        if(err == CoreApiConst::ErrorCode::Ok)
        {
            list << tlv;
        }
        else
        {
            if(err != CoreApiConst::ErrorCode::NoDataRequested) return err;
        }
    }
    if(regrep->type() == static_cast<quint8>(fdf::DocType::Registration))
    {
        rd = new RegDoc(nullptr);
    }
    else
    {
        rd = new ReRegDoc(nullptr);
    }
    if(!rd) return CoreApiConst::ErrorCode::UnknownFsError;
    rd->parseFromTlvList(list);
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readFsCounters(FsTotalType type, FsCounters &counters)
{
    counters.clean();
    FsReq req(fs::ReqCode::FsCounters, QByteArray(1, static_cast<char>(type)));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(60000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    return counters.parseFromFs(rep.data(), type) ?
                CoreApiConst::ErrorCode::Ok :
                CoreApiConst::ErrorCode::FsRepParsingError;
}

CoreApiConst::ErrorCode FsWrapper::readBStatus(BStatus &status)
{
    status = BStatus();
    FsReq req(fs::ReqCode::FsForLabelsStatus);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(5000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    if(rep.data().size() != 8) return CoreApiConst::ErrorCode::UnknownFsError;
    status.setCheckState(static_cast<BStatus::CheckState>(rep.data().at(0)));
    status.setNotificationState(static_cast<BStatus::NotificationState>(rep.data().at(1)));
    status.setFlags(BStatus::Flags(static_cast<quint8>(rep.data().at(2))));
    status.setResultsCount(static_cast<quint8>(rep.data().at(3)));
    status.setNotificatedCount(static_cast<quint8>(rep.data().at(4)));
    status.setWarning(static_cast<BStatus::OverflowWarning>(rep.data().at(5)));
    QDataStream ds(rep.data().mid(6));
    ds.setByteOrder(QDataStream::LittleEndian);
    quint16 c = 0;
    ds >> c;
    status.setNotications(c);
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readBStatus2(bool &canSkipB5)
{
    canSkipB5 = false;
    FsReq req(fs::ReqCode::FsForLabelsStatus, QByteArray(1, '\x00'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(5000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiConst::ErrorCode::Ok;
//        if(err == CoreApiConst::ErrorCode::UnknownFsCmd)
//        {
//        }
//        return err;
    }
    if(rep.data().size() != 1) return CoreApiConst::ErrorCode::UnknownFsError;
    canSkipB5 = (rep.data().at(0) == '\x01');
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::readLabelsTransport(std::optional<LabelsTransportState> &status)
{
    status.reset();
    FsReq req(fs::ReqCode::GetNotificationState);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(5000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        if(rep.data().size() != 13) return CoreApiConst::ErrorCode::UnknownFsError;
        LabelsTransportState st;
        st.setStatus(static_cast<LabelsTransportState::Status>(rep.data().at(0)));
        st.setMemFullPercent(static_cast<quint8>(rep.data().at(12)));
        QDataStream ds(rep.data().mid(1, 6));
        ds.setByteOrder(QDataStream::LittleEndian);
        quint16 b1 = 0;
        quint32 b2 = 0;
        ds >> b1 >> b2;
        st.setNotificationCount(b1);
        st.setFirstNotificationNumber(b2);
        st.setFirstNotificationDt(FsRep::parseDt(rep.data().mid(7, 5)));
        status = st;
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::writeTransportStaus(bool connected)
{
    FsReq req(fs::ReqCode::SendOfdConnectionState, QByteArray(1, connected ? '\x01' : '\x00' ));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::cancelMessageForOfd()
{
    FsReq req(fs::ReqCode::CancelOfdMsgReading);
    FsRep rep;
    return exec(req, rep, qMakePair(2000, 1000));
}

CoreApiConst::ErrorCode FsWrapper::readMessageForOfd(QByteArray &data)
{
    data.clear();
    FsReq req(fs::ReqCode::BeginOfdMsgReading);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(20000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        cancelMessageForOfd();
        return err;
    }
    quint16 len = 0;
    {
        QDataStream ds(rep.data_);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds >> len;
    }

    while(data.size() < static_cast<qint32>(len))
    {
        quint16 offcet = data.size(), length = qMin(len - offcet, 1024);
        QByteArray buf;
        QDataStream ds(&buf, QIODevice::WriteOnly);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds << offcet << length;
        FsReq rq(fs::ReqCode::ReadOfdMsgBlock, buf);
        FsRep rp;
        err = exec(rq, rp, qMakePair(3000, 1000));
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            cancelMessageForOfd();
            return err;
        }
        data.append(rp.data());
    }
    if(data.size() != len)
    {
        cancelMessageForOfd();
        return CoreApiConst::ErrorCode::UnknownFsError;
    }
    rep = FsRep();
    req = FsReq(fs::ReqCode::FinishOfdMsgReading);
    err = exec(req, rep, qMakePair(2000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::writeOfdTicket(const QByteArray &data, quint8 &fsAnswer,
                                                  quint8 &ofdAnswer, Tlv &ofdMsg, QByteArray &errDump)
{
    fsAnswer = 0;
    ofdAnswer = 0;
    ofdMsg.clean();
    errDump.clear();
    errDump.clear();
    FsReq req(fs::ReqCode::SendOfdReceipt, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(20000, 1000));
    switch (rep.repCode()) {
    case fs::RepCode::Ok:
    {
        if(rep.data().isEmpty()) return CoreApiConst::ErrorCode::UnknownFsError;
        ofdAnswer = rep.data()[0];
        if(rep.data().size() > 0) ofdMsg.setRawData(rep.data().mid(1));
        return CoreApiConst::ErrorCode::Ok;
    } break;
    case fs::RepCode::OfdMsgError:
    {
        if(rep.data().isEmpty()) return CoreApiConst::ErrorCode::UnknownFsError;
        fsAnswer = rep.data()[0];
        if(rep.data().size() > 0) ofdMsg.setRawData(rep.data().mid(1));
        return err;
    }break;
    default:
    {
        getFsLastErrors(errDump);
    } break;
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::startFiscalization(fs::RegType regType, fs::FFD ffd)
{
    FsReq req(regType, ffd);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(20000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return err;
    }
    return rep.repError();
}

CoreApiConst::ErrorCode FsWrapper::completeFiscalization(const QDateTime &dt, const Inn &userInn,
                                                         const RegNumber &regNumber, const fdf::Taxes &taxes,
                                                         const fs::RegFlags &flags, const fs::ExtRegFlags &extFlags,
                                                         const Inn &innOfd, qint32 &fd, quint32 &fiscCode,
                                                         fdf::Reasons *reasons)
{
    QByteArray data = FsReq::serializeDt(dt);
    data.append(userInn.data());
    data.append(regNumber.data());
    data.push_back(static_cast<char>(taxes));
    data.push_back(static_cast<char>(flags));
    data.push_back(static_cast<char>(extFlags));
    if(flags.testFlag(fs::RegFlag::OfflineMode))
    {
        data.append("000000000000", 12);
    }
    else
    {
        data.append(innOfd.data());
    }
    if(reasons)
    {
        QByteArray buf;
        QDataStream ds(&buf, QIODevice::WriteOnly);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds << static_cast<quint32>(*reasons);
        data.append(buf);
    }

    FsReq req(fs::ReqCode::CompleteRegistrationDoc2, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)return err;
    if(rep.repError() != CoreApiConst::ErrorCode::Ok)return err;
    return parsingResultToError(rep.getDocResult(fd, fiscCode));
}

CoreApiConst::ErrorCode FsWrapper::openCycleStart(const QDateTime &dt)
{
    FsReq req(fs::ReqCode::BeginCycleOpening, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::openCycleCommit(quint16 &cycle, qint32 &fd, quint32 &fiscCode)
{
    FsReq req(fs::ReqCode::CompleteCycleOpening);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        QDataStream ds(rep.data());
        ds.setByteOrder(QDataStream::LittleEndian);
        ds >> cycle >> fd >> fiscCode;
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::calcReportStart(const QDateTime &dt)
{
    FsReq req(fs::ReqCode::BeginPaymentStateReport, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::calcReportCommit(qint32 &fd, quint32 &fiscCode, quint32 &notSended, QDate &notSendedDt)
{
    FsReq req(fs::ReqCode::CompletePaymentStateReport);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        QDataStream ds(rep.data());
        ds.setByteOrder(QDataStream::LittleEndian);
        ds  >> fd >> fiscCode >> notSended;
        notSendedDt = FsRep::parseDate(rep.data().mid(12, 3));
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::closeCycleStart(const QDateTime &dt)
{
    FsReq req(fs::ReqCode::BeginCycleClosing, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::closeCycleCommit(quint16 &cycle, qint32 &fd, quint32 &fiscCode, fs::Warnings &warnings)
{
    FsReq req(fs::ReqCode::CloseCycle);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        QDataStream ds(rep.data());
        ds.setByteOrder(QDataStream::LittleEndian);
        quint8 buf = 0;
        ds  >> cycle>> fd >> fiscCode >> buf;
        warnings = fs::Warning(buf);
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::closeFsStart()
{
    FsReq req(fs::ReqCode::StartFsClosing);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::closeFsCommit(const QDateTime &dt, const RegNumber &rnm,
                                                 qint32 &fd, quint32 &fiscCode)
{
    FsReq req(fs::ReqCode::CloseFsFiscalMode, FsReq::serializeDt(dt) + rnm.data());
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(60000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        return parsingResultToError(rep.getDocResult(fd, fiscCode));
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::receiptStart(const QDateTime &dt)
{
    FsReq req(fs::ReqCode::BeginCheckCreating, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::correctionStart(const QDateTime &dt)
{
    FsReq req(fs::ReqCode::BeginCorrectionCheckCreating, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::receiptCommit(const QDateTime &dt, fdf::PaymentType pt, const FixNumber &amount, quint16 &receipt, qint32 &fd, quint32 &fiscalCode)
{
    receipt = 0;
    fd = 0;
    fiscalCode = 0;
    QByteArray buf = FsReq::serializeDt(dt);
    buf.push_back(static_cast<char>(pt));
    buf.append(FormatUtils::fixNumberToUint40(amount));
    FsReq req(fs::ReqCode::CompleteCheckCreating, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        if(rep.data().size() != 10) err = CoreApiConst::ErrorCode::UnknownFsError;
        else
        {
            QDataStream ds(rep.data());
            ds.setByteOrder(QDataStream::LittleEndian);
            ds >> receipt >> fd >> fiscalCode;
        }
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::getKeysAddress(bool test, fs::KeysStatus &ks,
                                                  QString &addr)
{
    FsReq req(fs::ReqCode::GetIngoForKeysUpdate, QByteArray(1, test ? '\x00' : '\x01'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(2000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(test)
    {
        if(rep.data().size() != 1) return CoreApiConst::ErrorCode::UnknownFsError;
        ks = static_cast<fs::KeysStatus>(rep.data().at(0));
    }
    else
    {
        QTextCodec *codec = QTextCodec::codecForName(fs::CODEC_NAME);
        addr = codec->toUnicode(rep.data().mid(0, rep.data().indexOf('\x00')) );
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::getKeysData(const QDateTime &dt, QByteArray &data)
{
    data.clear();
    FsReq req(fs::ReqCode::GetRequestForKeysUpdate, FsReq::serializeDt(dt));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(5000, 1000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        data = rep.data();
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::writeKeysData(const QByteArray &data, quint8 &fsAnswer,
                                                 quint16 &srvAns, QString &srvMsg)
{
    fsAnswer = 0;
    srvAns = 0;
    srvMsg.clear();
    FsReq req(fs::ReqCode::SendReplyForKeysUpdate, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(20000, 1000));
    switch (rep.repCode()) {
    case fs::RepCode::Ok:
    {
        return CoreApiConst::ErrorCode::Ok;
    } break;
    case fs::RepCode::OfdMsgError:
    {
        if(rep.data().isEmpty()) return CoreApiConst::ErrorCode::UnknownFsError;
        fsAnswer = rep.data()[0];
        return err;
    }break;
    case fs::RepCode::KeysServerFailure:
    {
        quint16 len = 0;
        QDataStream ds(rep.data());
        ds.setByteOrder(QDataStream::LittleEndian);
        ds >> srvAns << len;
        QTextCodec *codec = QTextCodec::codecForName(fs::CODEC_NAME);
        srvMsg = codec->toUnicode(rep.data().mid(4, len));
        return err;
    }break;
    case fs::RepCode::KeysServerUnknownError:
    {
        return err;
    }break;
    default: return err;
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::b1(fdf::LabelCodeType lct, const QByteArray &code,
                                      quint8 offcet91, quint8 offfcet92,
                                      fdf::LabelCodeCheckFsResult &r, fdf::LabelCodeCheckFsStatus &s)
{
    if(code.size() > 255) return CoreApiConst::ErrorCode::TooLongLabel;
    QByteArray buf;
    buf.push_back(static_cast<char>(lct));
    buf.push_back(static_cast<quint8>(code.size()));
    buf.append(code);
    if(lct == fdf::LabelCodeType::Code88Checked || lct == fdf::LabelCodeType::Code44Checked)
    {
        buf.push_back(offcet91);
        buf.push_back(offfcet92);
    }
    FsReq req(fs::ReqCode::SendLabelForFsCheck, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(rep.data().size() != 2) return CoreApiConst::ErrorCode::UnknownFsError;
    r = static_cast<fdf::LabelCodeCheckFsResult>(rep.data().at(0));
    s = static_cast<fdf::LabelCodeCheckFsStatus>(rep.data().at(1));
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::b1(const LabelCode &code,
                                      fdf::LabelCodeCheckFsResult &r,
                                      fdf::LabelCodeCheckFsStatus &s)
{
    return b1(code.lct2100(), code.preparedLabel(), code.idx91(), code.idx92(), r, s);
}

CoreApiConst::ErrorCode FsWrapper::d3(bool startSession, OfflineDSessionStatus &status)
{
    status = OfflineDSessionStatus();
    FsReq req(fs::ReqCode::StartUploadSession, QByteArray(1, startSession ? '\x01' : '\x00'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(!status.parseFs(rep.data())) return CoreApiConst::ErrorCode::UnknownFsError;
    return err;
}

CoreApiConst::ErrorCode FsWrapper::d4(bool startNext, quint16 &len, quint32 &num)
{
    len = 0;
    num = 0;
    FsReq req(fs::ReqCode::ReadNextNotification, QByteArray(1, startNext ? '\x01' : '\x00'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    QDataStream ds(rep.data());
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> len >> num;
    return err;
}

CoreApiConst::ErrorCode FsWrapper::d5(quint16 offcet, quint16 len, QByteArray &data, quint16 &invalidTag)
{
    invalidTag = 0;
    data.clear();
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << offcet << len;
    FsReq req(fs::ReqCode::ReadCurrentNotificationBlock, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err == CoreApiConst::ErrorCode::Ok) data = rep.data();
    else
    {
        if(rep.data().size() > 1 && err == CoreApiConst::ErrorCode::DeniedParams)
        {
            QDataStream ds(rep.data());
            ds.setByteOrder(QDataStream::LittleEndian);
            ds >> invalidTag;
        }
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::d6_0(quint16 &notConfirmed, quint32 &firstNotComfirmed)
{
    notConfirmed = 0;
    firstNotComfirmed = 0;
    FsReq req(fs::ReqCode::CommitNotificationUpload, QByteArray(1, '\x00'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    QDataStream ds(rep.data());
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> notConfirmed >> firstNotComfirmed;
    return err;
}

CoreApiConst::ErrorCode FsWrapper::d6_1(quint32 num, quint16 crc)
{
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << static_cast<quint8>(1) << num << crc;
    FsReq req(fs::ReqCode::CommitNotificationUpload, buf);
    FsRep rep;
    return exec(req, rep, qMakePair(10000, 2000));
}

CoreApiConst::ErrorCode FsWrapper::b2(bool saveResult, fdf::CheckLabelFlags &flags)
{

    FsReq req(fs::ReqCode::SaveLabelCheckResult, QByteArray(1, saveResult ? '\x01' : '\x00'));
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(saveResult && !rep.data().isEmpty())
    {
        flags = fdf::CheckLabelFlags(static_cast<quint8>(rep.data().at(0)));
    }
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::b2Cancel()
{
    fdf::CheckLabelFlags flags;
    return b2(false, flags);
}

CoreApiConst::ErrorCode FsWrapper::b3()
{
    FsReq req(fs::ReqCode::CleanLabelCheckResult);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::b5(const QDateTime &dt, const Tlv::Stlv &in, QByteArray &out)
{
    out.clear();
    QByteArray buf = FsReq::serializeDt(dt);
    for(const Tlv& t: in) buf.append(t.rawData());
    FsReq req(fs::ReqCode::CreateLabelRequest, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        out = rep.data();
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::b6(const QByteArray &data, fdf::CheckLabelFlags &flags, Tlv::Stlv &out)
{
    out.clear();
    FsReq req(fs::ReqCode::SendReplyAboutLabel, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        if(rep.data().isEmpty()) return CoreApiConst::ErrorCode::UnknownFsError;
        flags = fdf::CheckLabelFlags(rep.data().at(0));
        QDataStream ds(rep.data().mid(1));
        ds.setByteOrder(QDataStream::LittleEndian);
        // WhatcherDb dbc(WhatcherDb::CM_READ_WRITE);
        FsFullStatus st = IOSG::getFs();
        qint32 cycle = st.cycleIsOpen() ? st.cycle().cycle() : st.cycle().cycle() + 1;
        while(!ds.atEnd())
        {
            Tlv t;
            ds >> t;
            CoreRuntimeStorage store;
            QString fsnum = IOSG::getFs().fsNumber().trimmed();
            if(t.tag() == fdf::Tag::RequestProcessCodes && t.toByte())
            {
                store.update2113(fsnum, fdf::IncorrectNotificationsFlag::IcorrectRequest, cycle);
            }
            if(t.tag() == fdf::Tag::RequestProcessResult && (t.toByte() & 0xA) != 0xA)
            {
                store.update2112(fsnum, fdf::IncorrectLabelsFlag::IncorrectLabelResponse, cycle);
            }
            out << t;
        }
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::b7_1(const Tlv &t2007, quint16 &invalidTag)
{
    return b7(1, t2007.rawData(), invalidTag);
}

CoreApiConst::ErrorCode FsWrapper::b7_2(const Tlv &t1059, quint16 &invalidTag)
{
    return b7(2, t1059.rawData(), invalidTag);
}

CoreApiConst::ErrorCode FsWrapper::b7_3(const Tlv::Stlv &list, quint16 &invalidTag)
{
    QByteArray data;
    QDataStream ds(&data, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    for(const Tlv &t :list) ds << t;
    return b7(3, data, invalidTag);
}

CoreApiConst::ErrorCode FsWrapper::readNotification(QByteArray &data)
{

    data.clear();
    FsReq req(fs::ReqCode::StartNotificationReading);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(20000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        cancelNotificationReading();
        return err;
    }
    quint16 len = 0;
    {
        QDataStream ds(rep.data_);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds >> len;
    }

    while(data.size() < static_cast<qint32>(len))
    {
        quint16 offcet = data.size(), length = qMin(len - offcet, 1024);
        QByteArray buf;
        QDataStream ds(&buf, QIODevice::WriteOnly);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds << offcet << length;
        FsReq rq(fs::ReqCode::ReadNotificationBlock, buf);
        FsRep rp;
        err = exec(rq, rp, qMakePair(10000, 1000));
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            cancelNotificationReading();
            return err;
        }
        data.append(rp.data());
    }
    if(data.size() != len)
    {
        cancelNotificationReading();
        return CoreApiConst::ErrorCode::UnknownFsError;
    }
    rep = FsRep();
    req = FsReq(fs::ReqCode::FinishNotificationReading);
    err = exec(req, rep, qMakePair(2000, 1000));
    return err;
}

CoreApiConst::ErrorCode FsWrapper::cancelNotificationReading()
{
    FsReq req(fs::ReqCode::CancelNotificationReading);
    FsRep rep;
    return exec(req, rep, qMakePair(2000, 1000));
}

CoreApiConst::ErrorCode FsWrapper::writeNotificationTicket(const QByteArray &data, quint8 &ofdAnswer,
                                                           Tlv::Stlv &list)
{
    ofdAnswer = 0;
    list.clear();
    FsReq req(fs::ReqCode::SendNotificationTicket, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(20000, 1000));
    // WhatcherDb dbc(WhatcherDb::CM_READ_WRITE);
    FsFullStatus st = IOSG::getFs();
    switch (rep.repCode()) {
    case fs::RepCode::Ok:
    {
        if(rep.data().isEmpty()) return CoreApiConst::ErrorCode::UnknownFsError;
        QDataStream ds(rep.data());
        ds.setByteOrder(QDataStream::LittleEndian);
        while(!ds.atEnd())
        {
            Tlv t;
            ds >> t;
            CoreRuntimeStorage store;
            QString fsnum = IOSG::getFs().fsNumber().trimmed();
            if(t.tag() == fdf::Tag::NotificationProcessResult && t.toByte())
            {
                qint32 cycle = st.cycleIsOpen() ? st.cycle().cycle() : st.cycle().cycle() + 1;
                store.update2112(fsnum, fdf::IncorrectLabelsFlag::IncorrectLabelTicket, cycle);
            }
            if(t.tag() == fdf::Tag::NotificationsProcessCodes && t.toByte())
            {
                qint32 cycle = st.cycleIsOpen() ? st.cycle().cycle() : st.cycle().cycle() + 1;
                store.update2113(fsnum, fdf::IncorrectNotificationsFlag::IncorrectNotification, cycle);
            }
            list << t;
        }
        return CoreApiConst::ErrorCode::Ok;
    } break;
    case fs::RepCode::OfdMsgError:
    {
        if(rep.data().isEmpty()) return CoreApiConst::ErrorCode::UnknownFsError;
        ofdAnswer = rep.data().at(0);
        return err;
    }break;
    default: break;
    }



    return err;
}


CoreApiConst::ErrorCode FsWrapper::findDocument(qint32 docNumber, PFsArchiveReportBase &doc, const QStringList &header)
{
    doc = PFsArchiveReportBase();
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << docNumber;
    FsReq req(fs::ReqCode::FsDocByNumber, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));

    if(err == CoreApiConst::ErrorCode::Ok)
    {
        if(rep.data().size() <= 2) return CoreApiConst::ErrorCode::UnknownFsError;
        quint8 docType = static_cast<quint8>(rep.data()[0]);
        bool hasTicket = rep.data()[1];
        doc = FsArchiveReportBase::createDocument(docType, hasTicket, rep.data(), header);
        if(!doc)
        {
            return CoreApiConst::ErrorCode::FsRepParsingError;
        }
    }
    return err;
}

CoreApiConst::ErrorCode FsWrapper::findLastDocument(fdf::DocType type, qint32 from,
                                                    const FsFullStatus &status, PFsArchiveReportBase &doc)
{
    static const QSet<qint32> SUPPORTED = {
        static_cast<qint32>(fdf::DocType::Registration   ),
        static_cast<qint32>(fdf::DocType::ReRegistration ),
        static_cast<qint32>(fdf::DocType::CycleOpen      ),
        static_cast<qint32>(fdf::DocType::CalcReport     ),
        static_cast<qint32>(fdf::DocType::Receipt        ),
        static_cast<qint32>(fdf::DocType::Correction     ),
        static_cast<qint32>(fdf::DocType::Fosa           ),
        static_cast<qint32>(fdf::DocType::FosaCorrection ),
        static_cast<qint32>(fdf::DocType::CycleClose     ),
        static_cast<qint32>(fdf::DocType::FsClose        )
    };
    if(!status.isFiscalized()) return CoreApiConst::ErrorCode::NoDataRequested;
    if(!SUPPORTED.contains(static_cast<qint32>(type))) return CoreApiConst::ErrorCode::NoDataRequested;
    if(static_cast<qint32>(status.status().lastFd()) < from)
    {
        from = status.status().lastFd();
    }
    qint32 firstInvalidDoc = 0;

    switch (type) {
    case fdf::DocType::Registration   : firstInvalidDoc = 0; break;
    case fdf::DocType::ReRegistration : firstInvalidDoc = 1; break;
    case fdf::DocType::CycleOpen      : firstInvalidDoc = 1; break;
    case fdf::DocType::CalcReport     : firstInvalidDoc = 1; break;
    case fdf::DocType::Receipt        : firstInvalidDoc = 2; break;
    case fdf::DocType::Correction     : firstInvalidDoc = 2; break;
    case fdf::DocType::Fosa           : firstInvalidDoc = 2; break;
    case fdf::DocType::FosaCorrection : firstInvalidDoc = 2; break;
    case fdf::DocType::CycleClose     : firstInvalidDoc = 2; break;
    case fdf::DocType::FsClose        : firstInvalidDoc = 2; break;
    default:
        firstInvalidDoc = 0;
        break;
    }

    while (from > firstInvalidDoc)
    {
        CoreApiConst::ErrorCode err = findDocument(from, doc, QStringList());
        from --;
        if(err != CoreApiConst::ErrorCode::Ok) return err;
        if(doc && doc->type() == static_cast<quint8>(type)) return CoreApiConst::ErrorCode::Ok;
    }
    return CoreApiConst::ErrorCode::NoDataRequested;
}

CoreApiConst::ErrorCode FsWrapper::getOfdTicketInfo(qint32 docNumber, PFsArchiveOfdTicket &ticket)
{
    ticket = PFsArchiveOfdTicket();
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << docNumber;
    FsReq req(fs::ReqCode::FsDocReceiptByNumber, buf);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    ticket = PFsArchiveOfdTicket(new FsArchiveOfdTicket());
    ticket->setDt(FsRep::parseDt(rep.data().mid(0, 5)));
    ticket->setSign(rep.data().mid(5, 18));
    QDataStream repds(rep.data().mid(23));
    repds.setByteOrder(QDataStream::LittleEndian);
    repds >> docNumber;
    ticket->setFd(docNumber);
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::getNotSendedDocsCount(quint16 &cnt)
{
    cnt = 0;
    FsReq req(fs::ReqCode::OfflineDocsCount);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    QDataStream repds(rep.data().mid(23));
    repds.setByteOrder(QDataStream::LittleEndian);
    repds >> cnt;
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::findRegDoc(quint8 seq, PFsArchiveReportBase &doc, const QStringList &header)
{
    FsReq req(fs::ReqCode::FiscalisationSummary, seq > 0 ? QByteArray(1, static_cast<char>(seq)) : QByteArray());
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(30000, 1000));
    if(err != CoreApiConst::ErrorCode::Ok) return err;

    doc = FsArchiveReportBase::createRegDocument(rep.data(), header);


    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::getCycleDates(const FsFullStatus &status,
                                                 QDateTime &opened, QDateTime &closed)
{
    opened = QDateTime();
    closed = QDateTime();
    if(status.cycle().cycle() <= 0 || status.status().lastFd() <= 0) return CoreApiConst::ErrorCode::Ok;
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    qint32 from = status.status().lastFd();
    PFsArchiveReportBase doc;
    if(!status.cycle().opened())
    {
        err = findLastDocument(fdf::DocType::CycleClose, from, status, doc);
        if(err != CoreApiConst::ErrorCode::Ok) return err;
        if(doc)
        {
            closed = doc->dt();
            from = static_cast<qint32>(doc->docNumber()) - 1;
        }
    }
    from = from - status.cycle().receipt();
    err = findLastDocument(fdf::DocType::CycleOpen, from, status, doc);
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(doc)
    {
        opened = doc->dt();
    }
    return err;
}


FsOfdResponse FsWrapper::ofdCmd(const OfdFsCmd &cmd)
{
    FsOfdResponse resp(cmd);
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    bool needReadErrorDump = false;
    switch (cmd.cmd()) {
    case OfdFsCmd::Code::TransfereStatus       :
    {
        std::optional<FsTransportState> ts;
        err = getFsTransport(ts);
        needReadErrorDump = (err == CoreApiConst::ErrorCode::FsError);
        resp.setTransportState(ts.value_or(FsTransportState()));

    }break;
    case OfdFsCmd::Code::SendOfdConnectionState:
    {
        err = writeTransportStaus(cmd.ofdConnected());
        needReadErrorDump = (err == CoreApiConst::ErrorCode::FsError);
    }break;
    case OfdFsCmd::Code::ReadOfdMsg            :
    {
        QByteArray data;
        err = readMessageForOfd(data);
        needReadErrorDump = (err == CoreApiConst::ErrorCode::FsError);
        resp.setOfdMessage(data);
    }break;
    case OfdFsCmd::Code::CancelOfdMsgReading   :
    {
        err = cancelMessageForOfd();
    }break;
    case OfdFsCmd::Code::SendOfdReceipt        :
    {
        quint8 fsAnswer = 0, ofdAnswer = 0;
        Tlv ofdMsg;
        QByteArray errDump;
        err = writeOfdTicket(cmd.ofdTicket(), fsAnswer, ofdAnswer, ofdMsg, errDump);
        OfdTicketResult r;
        r.setFsCode(static_cast<fdf::FsTicketAnswerCode>(fsAnswer));
        r.setOfdCode(static_cast<fdf::OfdAnswerCode>(ofdAnswer));
        r.setOfdMessage(ofdMsg);
        resp.setTicketResult(r);
        resp.setErrorDump(errDump);
    }break;

    case OfdFsCmd::Code::ReadNotification        :
    {
        QByteArray data;
        err = readNotification(data);
        needReadErrorDump = (err == CoreApiConst::ErrorCode::FsError);
        resp.setOfdMessage(data);
    }break;
    case OfdFsCmd::Code::CancelNofification      :
    {
        err = cancelNotificationReading();
    }break;
    case OfdFsCmd::Code::WriteNotificationTicket :
    {
        quint8 ofdAnswer = 0;
        Tlv::Stlv list;
        err = writeNotificationTicket(cmd.ofdTicket(), ofdAnswer, list);
        OfdTicketResult r;
        r.setOfdCode(static_cast<fdf::OfdAnswerCode>(ofdAnswer));
        r.setOismProps(list);
        resp.setTicketResult(r);
    }break;
    default:
        err = CoreApiConst::ErrorCode::UnknownFsCmd;
        break;
    }
    resp.setError(err);

    if(needReadErrorDump)
    {
        QByteArray d;
        getFsLastErrors(d);
        resp.setErrorDump(d);
    }
    return resp;
}

CoreApiConst::ErrorCode FsWrapper::exec(const FsReq &req, FsRep &rep,
                                        const FsRwTimeout &tout)
{
    if(!api_->isOpen() && !api_->open())
    {
        return CoreApiConst::ErrorCode::FsIsOffline;
    }
    bool res  = api_->exec(req, rep, tout);
    if(!res)
    {
        return CoreApiConst::ErrorCode::FsIsOffline;
    }
    if(rep.parsingResult() != fs::RepParsingRes::Ok)
    {
        return CoreApiConst::ErrorCode::FsRepParsingError;
    }
    return rep.repError();
}

CoreApiConst::ErrorCode FsWrapper::parsingResultToError(fs::RepParsingRes res)
{
    if(res != fs::RepParsingRes::Ok)
    {
        return CoreApiConst::ErrorCode::FsRepParsingError;
    }

    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode FsWrapper::b7(quint8 mode, QByteArray data, quint16 &invalidTag)
{
    invalidTag = 0;
    data.push_front(mode);
    FsReq req(fs::ReqCode::SendFiscalDataForLabels, data);
    FsRep rep;
    CoreApiConst::ErrorCode err = exec(req, rep, qMakePair(10000, 2000));

    if(err != CoreApiConst::ErrorCode::Ok)
    {
        if(rep.data().size() > 1 && err == CoreApiConst::ErrorCode::DeniedParams)
        {
            QDataStream ds(rep.data());
            ds.setByteOrder(QDataStream::LittleEndian);
            ds >> invalidTag;
        }
        QDataStream ds(data.mid(1));
        ds.setByteOrder(QDataStream::LittleEndian);
        Tlv buf;
        QStringList sl;
        while(!ds.atEnd())
        {
            ds >> buf;
            sl << QString::number(static_cast<qint32>(buf.tag())) << QString::fromLatin1(buf.value().toHex());
            if(buf.tag() == fdf::Tag::ReceiptItem || buf.tag() == fdf::Tag::LabledProductInfo)
            {
                Tlv::Stlv list = buf.toStlv();
                for(const Tlv &t: list)
                {
                    sl << "    " << QString::number(static_cast<qint32>(t.tag())) << QString::fromLatin1(t.value().toHex());
                }
            }
        }
    }
    return err;
}
